/**
 * Add Presentation Controller
 *
 * Handles PDF presentation upload to Azure Blob Storage
 * - Validates PDF file upload
 * - Uploads to presentation-materials container with pdf/ prefix
 * - Saves presentation metadata to database
 */

const database = require("@database");
const utils = require("@utils");
const azureStorage = require("@utils/azure.storage.util");
const path = require("path");

const addPresentation = async (req, res) => {
  try {
    // Get user UGID from token
    const { ugid } = req.user;

    // Validate file upload
    if (!req.file) {
      return utils.responseSnippet(res, 400, "No file uploaded", null);
    }

    const file = req.file;

    // Get presentation name from request body or use original filename
    const presentationName = req.body.name || path.parse(file.originalname).name;

    // Create unique blob name with timestamp
    const timestamp = Date.now();
    const blobName = `pdf/${ugid}-${timestamp}.pdf`;
    const containerName = "presentation-materials";

    // Upload PDF to Azure Blob Storage
    // Note: Only store ASCII-safe metadata in Azure
    // Full presentation details (including names with special characters) are stored in database
    const uploadResult = await azureStorage.uploadFile(
      containerName,
      blobName,
      file.buffer,
      {
        contentType: file.mimetype,
        metadata: {
          ugid: ugid,
          uploadedAt: new Date().toISOString(),
        },
      }
    );

    if (!uploadResult.success) {
      return utils.responseSnippet(
        res,
        500,
        `Failed to upload presentation: ${uploadResult.message}`,
        null
      );
    }

    // Save presentation metadata to database
    const [presentationId] = await database("presentation_pdf").insert({
      ugid,
      presentation_name: presentationName,
      file_blob_key: blobName, // Store blob key (e.g., "pdf/user123-1234567890.pdf")
    });

    // Fetch the created presentation
    const presentation = await database("presentation_pdf")
      .where({ id: presentationId })
      .select("id", "presentation_name")
      .first();

    return utils.responseSnippet(
      res,
      200,
      "Presentation uploaded successfully",
      presentation
    );
  } catch (error) {
    console.error("Add presentation error:", error);
    return utils.catchErrorHandlerSnippet(res, error, __dirname);
  }
};

module.exports = addPresentation;
