/**
 * Rename Session Controller
 *
 * Renames a presentation session
 * - Validates session exists
 * - Checks authorization (user can only rename their own sessions)
 * - Updates session_name field
 */

const database = require("@database");
const utils = require("@utils");

const renameSession = async (req, res) => {
  try {
    // Get user UGID from token
    const { ugid } = req.user;

    // Get session ID from params and name from body (already validated by Joi)
    const id = req.params.id;
    const { session_name } = req.body;

    // Check if session exists
    const session = await database("presentation_session")
      .where({ id })
      .first();

    if (!session) {
      return utils.responseSnippet(res, 404, "Session not found", null);
    }

    // Authorization check: User can only rename their own sessions
    if (session.ugid !== ugid) {
      return utils.responseSnippet(
        res,
        403,
        "You do not have permission to rename this session",
        null
      );
    }

    // Update session name
    await database("presentation_session")
      .where({ id })
      .update({
        session_name,
        // updated_at: new Date(), // REMOVED
      });

    return utils.responseSnippet(
      res,
      200,
      "Session renamed successfully",
      { id, session_name }
    );
  } catch (error) {
    console.error("Rename session error:", error);
    return utils.catchErrorHandlerSnippet(res, error, __dirname);
  }
};

module.exports = renameSession;
