/**
 * Update Session Data Controller
 *
 * Updates session analysis data after processing
 * - Validates session exists
 * - Checks authorization (user can only update their own sessions)
 * - Updates transcript, audio analysis, scores, and status
 * - Used by analysis service after processing is complete
 */

const database = require("@database");
const utils = require("@utils");

const updateSessionData = async (req, res) => {
  try {
    // Get user UGID from token
    const { ugid } = req.user;

    // Get session ID from params and update data from body (already validated by Joi)
    const id = req.params.id;
    const updateData = req.body;

    // Check if session exists
    const session = await database("presentation_session")
      .where({ id })
      .first();

    if (!session) {
      return utils.responseSnippet(res, 404, "Session not found", null);
    }

    // Authorization check: User can only update their own sessions
    if (session.ugid !== ugid) {
      return utils.responseSnippet(
        res,
        403,
        "You do not have permission to update this session",
        null
      );
    }

    // Prepare update object (only include fields that were provided)
    const fieldsToUpdate = {};

    // Add provided fields to update
    const allowedFields = [
      "transcript_text",
      "transcript_words",
      "transcript_sentences",
      "transcribed_segments",
      "filler_words",
      "key_phrases",
      "key_words",
      "words_per_minute",
      "sentiment_positive_score",
      "sentiment_negative_score",
      "sentiment_neutral_score",
      "audio_pitch_data",
      "audio_volume_data",
      "audio_pauses_data",
      // "audio_timed_averages", // REMOVED
      "avg_pitch",
      "avg_volume",
      "avg_pause_time",
      "pauses_per_minute",
      "total_pause_time",
      "speech_speed_score",
      "pauses_score",
      "repetition_score",
      "filler_words_score",
      "speaking_clearly_score",
      "overall_score",
      // "analysis_status", // REMOVED
    ];

    allowedFields.forEach((field) => {
      if (updateData[field] !== undefined) {
        fieldsToUpdate[field] = updateData[field];
      }
    });

    // Add updated_at timestamp - REMOVED (column removed)
    // fieldsToUpdate.updated_at = new Date();

    // Update session
    await database("presentation_session").where({ id }).update(fieldsToUpdate);

    return utils.responseSnippet(
      res,
      200,
      "Session data updated successfully",
      { id, updated_fields: Object.keys(fieldsToUpdate) }
    );
  } catch (error) {
    console.error("Update session data error:", error);
    return utils.catchErrorHandlerSnippet(res, error, __dirname);
  }
};

module.exports = updateSessionData;
