/**
 * Score Calculation Utility
 *
 * Ported from C# ScoreCalculationService
 * Provides methods for calculating presentation session scores based on various metrics
 *
 * Score Range: 0 (worst) to 10 (best)
 * Uses trapezoidal scoring algorithm with optimal ranges
 */

const WORST_SCORE = 0;
const BEST_SCORE = 10;

/**
 * Value Range for score calculation
 * Defines the trapezoidal range for scoring
 *
 * @typedef {Object} ValueRange
 * @property {number} outerMin - Below this value = worst score (0)
 * @property {number} bestMin - From outerMin to bestMin = linear increase 0→10
 * @property {number} bestMax - From bestMin to bestMax = best score (10)
 * @property {number} outerMax - From bestMax to outerMax = linear decrease 10→0, above = worst score (0)
 */
class ValueRange {
  constructor(outerMin, bestMin, bestMax, outerMax) {
    this.outerMin = outerMin;
    this.bestMin = bestMin;
    this.bestMax = bestMax;
    this.outerMax = outerMax;
  }
}

/**
 * Linear equation: y = mx + t
 * @param {number} x - Input value
 * @param {number} m - Gradient (slope)
 * @param {number} t - Y-intercept
 * @returns {number} Result
 */
function linear(x, m, t) {
  return m * x + t;
}

/**
 * Maps an input value from one range to another linearly
 * @param {number} outputMin - Minimum output value
 * @param {number} outputMax - Maximum output value
 * @param {number} input - Input value
 * @param {number} inputMin - Minimum input value
 * @param {number} inputMax - Maximum input value
 * @param {boolean} outputBorderedToMax - Clamp to max (default true)
 * @param {boolean} outputBorderedToMin - Clamp to min (default true)
 * @returns {number} Mapped value
 */
function linearRange(
  outputMin,
  outputMax,
  input,
  inputMin,
  inputMax,
  outputBorderedToMax = true,
  outputBorderedToMin = true
) {
  const inputRange = inputMax - inputMin;

  if (outputBorderedToMax && input > inputMax) {
    return outputMax;
  } else if (outputBorderedToMin && input < inputMin) {
    return outputMin;
  } else {
    const gradient = (outputMax - outputMin) / inputRange;
    return linear(input - inputMin, gradient, outputMin);
  }
}

/**
 * Calculates score based on value and range using trapezoidal algorithm
 *
 * Scoring Logic:
 * - value < outerMin → worst score (0)
 * - outerMin ≤ value < bestMin → linear increase 0→10
 * - bestMin ≤ value ≤ bestMax → best score (10)
 * - bestMax < value < outerMax → linear decrease 10→0
 * - value ≥ outerMax → worst score (0)
 *
 * @param {ValueRange} valueRange - The scoring range
 * @param {number} value - The value to score
 * @returns {number} Score from 0 to 10
 */
function getScore(valueRange, value) {
  if (value < valueRange.outerMin) {
    return WORST_SCORE;
  } else if (value < valueRange.bestMin) {
    return linearRange(
      WORST_SCORE,
      BEST_SCORE,
      value,
      valueRange.outerMin,
      valueRange.bestMin
    );
  } else if (value <= valueRange.bestMax) {
    return BEST_SCORE;
  } else if (value < valueRange.outerMax) {
    return linearRange(
      BEST_SCORE,
      WORST_SCORE,
      value,
      valueRange.bestMax,
      valueRange.outerMax
    );
  } else {
    // value >= outerMax
    return WORST_SCORE;
  }
}

/**
 * Calculate all session scores based on analysis data
 *
 * @param {Object} analysisData - Session analysis data
 * @param {number} analysisData.wordsPerMinute - WPM from transcription
 * @param {number} analysisData.pausesPerMinute - Pauses per minute from audio analysis
 * @param {number} analysisData.fillerWordsCount - Total filler words detected
 * @param {number} analysisData.sessionLengthMinutes - Session duration in minutes
 * @param {number} analysisData.averageConfidence - Average transcription confidence (0-1)
 * @param {Array} analysisData.lookupWords - Array of word occurrences for repetition detection
 * @returns {Object} All calculated scores
 */
function calculateAllScores(analysisData) {
  const {
    wordsPerMinute = 0,
    pausesPerMinute = 0,
    fillerWordsCount = 0,
    sessionLengthMinutes = 1,
    averageConfidence = 0,
    lookupWords = [],
  } = analysisData;

  // ============================================================================
  // 1. SPEECH SPEED SCORE
  // ============================================================================
  // Optimal range: 95-145 WPM
  // Too slow (<70) or too fast (>180) = 0
  // Extended upper limit from 160 to 180 to be more forgiving with fast speakers
  const speechSpeedScore = getScore(
    new ValueRange(70, 95, 145, 180),
    wordsPerMinute
  );

  // ============================================================================
  // 2. FILLER WORDS SCORE
  // ============================================================================
  // Calculate filler words per minute
  const fillerWordsPerMinute = fillerWordsCount / sessionLengthMinutes;
  // Optimal: 0-2 filler words per minute
  const fillerWordsScore = getScore(
    new ValueRange(0, 0, 2, 11),
    fillerWordsPerMinute
  );

  // ============================================================================
  // 3. PAUSES SCORE
  // ============================================================================
  // Optimal range: 2-6 pauses per minute
  // Too few (<2) or too many (>6) is bad
  const pausesScore = getScore(
    new ValueRange(0, 2, 6, 9),
    pausesPerMinute
  );

  // ============================================================================
  // 4. SPEAKING CLEARLY SCORE
  // ============================================================================
  // Based on Azure transcription confidence (0-1)
  // Higher confidence = clearer speech
  const speakingClearlyScore = getScore(
    new ValueRange(0, 1, 1, 1),
    averageConfidence
  );

  // ============================================================================
  // 5. REPETITION SCORE
  // ============================================================================
  // Find non-filler words repeated more than 3 times per minute
  // lookupWords should be: [{ word: string, occurrences: number, type: 'filler'|'default' }]
  const wordsForRepetition = lookupWords
    .filter((w) => w.type !== 'filler' && w.type !== 'FillerWord')
    .map((w) => w.occurrences / sessionLengthMinutes)
    .filter((occurrencesPerMin) => occurrencesPerMin > 3);

  let repetitionScore = BEST_SCORE; // Default to best score if no repetitions

  if (wordsForRepetition.length > 0) {
    // Calculate average score for all repeated words
    const repetitionScores = wordsForRepetition.map((occurrencesPerMin) =>
      getScore(new ValueRange(0, 0, 3, 12), occurrencesPerMin)
    );
    repetitionScore =
      repetitionScores.reduce((sum, score) => sum + score, 0) /
      repetitionScores.length;
  }

  // ============================================================================
  // 6. OVERALL SCORE (WEIGHTED AVERAGE)
  // ============================================================================
  // Scores ≤ 3 are counted 3 times to emphasize problems
  let overallSum = 0;
  let divisor = 0;

  const scores = [
    speechSpeedScore,
    fillerWordsScore,
    pausesScore,
    speakingClearlyScore,
    repetitionScore,
  ];

  scores.forEach((score) => {
    if (score <= 3) {
      // Poor score - count it 3 times
      overallSum += 3 * score;
      divisor += 3;
    } else {
      // Good score - count it once
      overallSum += score;
      divisor += 1;
    }
  });

  const overallScore = divisor > 0 ? overallSum / divisor : 0;

  // ============================================================================
  // RETURN ALL SCORES
  // ============================================================================
  return {
    speechSpeedScore: Math.round(speechSpeedScore * 100) / 100,
    fillerWordsScore: Math.round(fillerWordsScore * 100) / 100,
    pausesScore: Math.round(pausesScore * 100) / 100,
    speakingClearlyScore: Math.round(speakingClearlyScore * 100) / 100,
    repetitionScore: Math.round(repetitionScore * 100) / 100,
    overallScore: Math.round(overallScore * 100) / 100,
  };
}

module.exports = {
  ValueRange,
  getScore,
  calculateAllScores,
  linear,
  linearRange,
};
