/**
 * Speech Recognition WebSocket Handler
 *
 * Handles WebSocket connections for real-time speech-to-text transcription
 * Features:
 * - JWT authentication
 * - Real-time audio streaming
 * - Interim and final results forwarding
 * - Session management
 * - Error handling
 */

const jwt = require('jsonwebtoken');
const config = require('@config');
const AzureSpeechService = require('./azureSpeechService');

// Active sessions map: connectionId -> { service, user, language }
const activeSessions = new Map();

/**
 * Setup speech recognition WebSocket handler
 */
function setupSpeechWebSocket(wss) {
  console.log('🎙️  [SpeechWebSocket] Setting up speech recognition WebSocket handler');

  wss.on('connection', (socket, request) => {
    console.log('🔌 [SpeechWebSocket] New WebSocket connection attempt');

    // Parse URL to check path and extract token
    const url = new URL(request.url, `ws://${request.headers.host}`);
    const pathname = url.pathname;

    // Only handle /ws/speech path
    if (!pathname.startsWith('/ws/speech')) {
      console.log('⏭️  [SpeechWebSocket] Ignoring non-speech connection:', pathname);
      return;
    }

    console.log('🎯 [SpeechWebSocket] Speech WebSocket connection on path:', pathname);

    // Extract and verify JWT token
    const token = url.searchParams.get('token');
    let user = null;

    if (!token) {
      console.error('❌ [SpeechWebSocket] No token provided');
      socket.send(JSON.stringify({
        type: 'error',
        message: 'Authentication token required'
      }));
      socket.close(1008, 'Authentication token required');
      return;
    }

    try {
      // Verify JWT token
      const jwtSecret = config.session.secret || process.env.JWT_SECRET_KEY;
      user = jwt.verify(token, jwtSecret);
      console.log('✅ [SpeechWebSocket] Token verified for user:', user.ugid || user.id);
    } catch (error) {
      console.error('❌ [SpeechWebSocket] Token verification failed:', error.message);
      socket.send(JSON.stringify({
        type: 'error',
        message: 'Invalid authentication token'
      }));
      socket.close(1008, 'Invalid token');
      return;
    }

    // Create unique connection ID
    const connectionId = `${user.ugid || user.id}_${Date.now()}`;
    console.log('🆔 [SpeechWebSocket] Connection ID:', connectionId);

    // Session state
    let speechService = null;
    let language = 'de-DE'; // Default language
    let isRecognizing = false;

    /**
     * Handle incoming messages from client
     */
    socket.on('message', async (message) => {
      try {
        const data = JSON.parse(message.toString());
        console.log('📩 [SpeechWebSocket] Received message:', {
          type: data.type,
          connectionId
        });

        switch (data.type) {
          case 'start':
            await handleStart(data);
            break;

          case 'audio':
            await handleAudio(data);
            break;

          case 'stop':
            await handleStop();
            break;

          case 'ping':
            socket.send(JSON.stringify({
              type: 'pong',
              timestamp: Date.now()
            }));
            break;

          default:
            console.warn('⚠️  [SpeechWebSocket] Unknown message type:', data.type);
            socket.send(JSON.stringify({
              type: 'error',
              message: `Unknown message type: ${data.type}`
            }));
        }
      } catch (error) {
        console.error('❌ [SpeechWebSocket] Message handling error:', error);
        socket.send(JSON.stringify({
          type: 'error',
          message: 'Failed to process message',
          details: error.message
        }));
      }
    });

    /**
     * Handle START command - initialize speech recognition
     */
    async function handleStart(data) {
      console.log('▶️  [SpeechWebSocket] START command received');

      if (isRecognizing) {
        console.warn('⚠️  [SpeechWebSocket] Already recognizing, stopping first');
        await handleStop();
      }

      try {
        // Get language from client or use default
        language = data.language || 'de-DE';
        console.log('🌍 [SpeechWebSocket] Language:', language);

        // Create new speech service
        speechService = new AzureSpeechService(language);

        // Setup with callbacks
        speechService.setup({
          onInterimResult: (text) => {
            console.log('📝 [SpeechWebSocket] Interim result, sending to client');
            socket.send(JSON.stringify({
              type: 'interim',
              text: text,
              timestamp: Date.now()
            }));
          },

          onFinalResult: (newText, fullText, segment) => {
            console.log('✅ [SpeechWebSocket] Final result, sending to client');
            socket.send(JSON.stringify({
              type: 'final',
              text: newText,
              fullText: fullText,
              segment: segment,
              timestamp: Date.now()
            }));
          },

          onError: (error) => {
            console.error('❌ [SpeechWebSocket] Speech service error:', error);
            socket.send(JSON.stringify({
              type: 'error',
              message: 'Speech recognition error',
              details: error.message
            }));

            // Cleanup on error
            isRecognizing = false;
            if (activeSessions.has(connectionId)) {
              activeSessions.delete(connectionId);
            }
          }
        });

        // Start recognition
        const started = await speechService.start();

        if (started) {
          isRecognizing = true;

          // Store session
          activeSessions.set(connectionId, {
            service: speechService,
            user: user,
            language: language,
            startTime: Date.now()
          });

          console.log('✅ [SpeechWebSocket] Recognition started, active sessions:', activeSessions.size);

          // Send success response
          socket.send(JSON.stringify({
            type: 'started',
            language: language,
            timestamp: Date.now()
          }));
        } else {
          throw new Error('Failed to start speech recognition');
        }
      } catch (error) {
        console.error('❌ [SpeechWebSocket] START error:', error);
        socket.send(JSON.stringify({
          type: 'error',
          message: 'Failed to start speech recognition',
          details: error.message
        }));
        isRecognizing = false;
      }
    }

    /**
     * Handle AUDIO command - stream audio data to speech service
     */
    async function handleAudio(data) {
      if (!isRecognizing || !speechService) {
        console.warn('⚠️  [SpeechWebSocket] Received audio but not recognizing');
        return;
      }

      try {
        // Get audio data (should be Int16Array sent as array from frontend)
        const audioData = data.audio;

        if (!audioData) {
          console.warn('⚠️  [SpeechWebSocket] No audio data in message');
          return;
        }

        // Convert to buffer based on data type
        let buffer;

        if (Buffer.isBuffer(audioData)) {
          // Already a buffer
          buffer = audioData;
        } else if (Array.isArray(audioData)) {
          // Array of Int16 values from frontend - convert to buffer
          // Each Int16 is 2 bytes
          buffer = Buffer.alloc(audioData.length * 2);
          for (let i = 0; i < audioData.length; i++) {
            buffer.writeInt16LE(audioData[i], i * 2);
          }
        } else if (typeof audioData === 'string') {
          // Base64 encoded string
          buffer = Buffer.from(audioData, 'base64');
        } else {
          console.error('❌ [SpeechWebSocket] Unknown audio data type:', typeof audioData);
          return;
        }

        // Send to speech service
        speechService.sendAudioChunk(buffer);

        // Log periodically (not every chunk)
        if (Math.random() < 0.005) { // 0.5% of chunks
          console.log('🎵 [SpeechWebSocket] Audio chunk processed:', buffer.length, 'bytes');
        }
      } catch (error) {
        console.error('❌ [SpeechWebSocket] AUDIO error:', error);
        socket.send(JSON.stringify({
          type: 'error',
          message: 'Failed to process audio',
          details: error.message
        }));
      }
    }

    /**
     * Handle STOP command - stop recognition and send final results
     */
    async function handleStop() {
      console.log('⏹️  [SpeechWebSocket] STOP command received');

      if (!isRecognizing || !speechService) {
        console.warn('⚠️  [SpeechWebSocket] Not recognizing, nothing to stop');
        socket.send(JSON.stringify({
          type: 'stopped',
          message: 'No active recognition session'
        }));
        return;
      }

      try {
        // Stop speech service and get results
        const results = await speechService.stop();

        console.log('📊 [SpeechWebSocket] Sending final results to client:', {
          wordCount: results.wordCount,
          duration: results.duration.toFixed(2) + 's',
          wpm: results.wpm,
          pauses: results.pausesMade
        });

        // Send final results to client
        socket.send(JSON.stringify({
          type: 'stopped',
          results: results,
          timestamp: Date.now()
        }));

        // Cleanup
        isRecognizing = false;
        speechService = null;

        if (activeSessions.has(connectionId)) {
          activeSessions.delete(connectionId);
          console.log('✅ [SpeechWebSocket] Session removed, active sessions:', activeSessions.size);
        }
      } catch (error) {
        console.error('❌ [SpeechWebSocket] STOP error:', error);
        socket.send(JSON.stringify({
          type: 'error',
          message: 'Failed to stop speech recognition',
          details: error.message
        }));

        // Cleanup anyway
        isRecognizing = false;
        speechService = null;
        if (activeSessions.has(connectionId)) {
          activeSessions.delete(connectionId);
        }
      }
    }

    /**
     * Handle connection errors
     */
    socket.on('error', (error) => {
      console.error('❌ [SpeechWebSocket] Socket error:', {
        connectionId,
        error: error.message
      });

      // Cleanup on error
      if (isRecognizing && speechService) {
        speechService.stop().catch(err => {
          console.error('❌ [SpeechWebSocket] Error stopping service on socket error:', err);
        });
      }

      isRecognizing = false;
      speechService = null;

      if (activeSessions.has(connectionId)) {
        activeSessions.delete(connectionId);
      }

      socket.terminate();
    });

    /**
     * Handle connection close
     */
    socket.on('close', async (code, reason) => {
      console.log('🔌 [SpeechWebSocket] Connection closed:', {
        connectionId,
        code,
        reason: reason.toString()
      });

      // Cleanup on close
      if (isRecognizing && speechService) {
        try {
          await speechService.stop();
        } catch (error) {
          console.error('❌ [SpeechWebSocket] Error stopping service on close:', error);
        }
      }

      isRecognizing = false;
      speechService = null;

      if (activeSessions.has(connectionId)) {
        activeSessions.delete(connectionId);
        console.log('✅ [SpeechWebSocket] Session cleaned up, active sessions:', activeSessions.size);
      }
    });

    console.log('✅ [SpeechWebSocket] Speech connection established for user:', user.ugid || user.id);
  });

  console.log('✅ [SpeechWebSocket] Speech WebSocket handler ready');
}

/**
 * Get active sessions count (for monitoring)
 */
function getActiveSessions() {
  return {
    count: activeSessions.size,
    sessions: Array.from(activeSessions.entries()).map(([id, session]) => ({
      connectionId: id,
      user: session.user.ugid || session.user.id,
      language: session.language,
      duration: Date.now() - session.startTime
    }))
  };
}

module.exports = {
  setupSpeechWebSocket,
  getActiveSessions
};
