/**
 * Analyze Audio Data in Database
 * Check for non-zero values in pitch and volume data
 */

require('module-alias/register');
const database = require('@database');

async function analyzeAudioData() {
  try {
    // Get most recent sessions
    const sessions = await database('presentation_session')
      .select('id', 'session_name', 'recording_date', 'audio_pitch_data', 'audio_volume_data', 'avg_pitch', 'avg_volume')
      .whereNotNull('audio_pitch_data')
      .whereNotNull('audio_volume_data')
      .orderBy('recording_date', 'desc')
      .limit(5);

    console.log('\n📊 Recent Sessions Audio Data Analysis:\n');
    console.log('=' .repeat(60));

    for (const session of sessions) {
      console.log('\n📁 Session:', session.session_name);
      console.log('   ID:', session.id);
      console.log('   Date:', new Date(session.recording_date).toLocaleString());

      // Analyze pitch data
      let pitchInfo = {
        total: 0,
        nonZero: 0,
        firstNonZero: null,
        avgFromDB: session.avg_pitch
      };

      if (session.audio_pitch_data) {
        try {
          const pitchData = JSON.parse(session.audio_pitch_data);
          pitchInfo.total = pitchData.length;
          pitchInfo.nonZero = pitchData.filter(p => p.value > 0).length;
          pitchInfo.firstNonZero = pitchData.find(p => p.value > 0);

          // Calculate actual average from data
          if (pitchInfo.nonZero > 0) {
            const sum = pitchData.reduce((acc, p) => acc + (p.value || 0), 0);
            pitchInfo.calculatedAvg = Math.round(sum / pitchData.length);
          }
        } catch(e) {
          console.error('   ❌ Error parsing pitch data:', e.message);
        }
      }

      // Analyze volume data
      let volumeInfo = {
        total: 0,
        nonZero: 0,
        firstNonZero: null,
        avgFromDB: session.avg_volume
      };

      if (session.audio_volume_data) {
        try {
          const volumeData = JSON.parse(session.audio_volume_data);
          volumeInfo.total = volumeData.length;
          volumeInfo.nonZero = volumeData.filter(v => v.value > 0).length;
          volumeInfo.firstNonZero = volumeData.find(v => v.value > 0);

          // Calculate actual average from data
          if (volumeInfo.nonZero > 0) {
            const sum = volumeData.reduce((acc, v) => acc + (v.value || 0), 0);
            volumeInfo.calculatedAvg = Math.round(sum / volumeData.length);
          }
        } catch(e) {
          console.error('   ❌ Error parsing volume data:', e.message);
        }
      }

      // Display results
      console.log('\n   🎵 Pitch Data:');
      console.log('      Total samples:', pitchInfo.total);
      console.log('      Non-zero samples:', pitchInfo.nonZero,
                  `(${pitchInfo.total > 0 ? Math.round(pitchInfo.nonZero/pitchInfo.total*100) : 0}%)`);
      console.log('      DB Average:', pitchInfo.avgFromDB || 'N/A');
      if (pitchInfo.calculatedAvg) {
        console.log('      Calculated Average:', pitchInfo.calculatedAvg);
      }
      if (pitchInfo.firstNonZero) {
        console.log('      First non-zero:', JSON.stringify(pitchInfo.firstNonZero));
      } else {
        console.log('      ⚠️  All values are zero!');
      }

      console.log('\n   🔊 Volume Data:');
      console.log('      Total samples:', volumeInfo.total);
      console.log('      Non-zero samples:', volumeInfo.nonZero,
                  `(${volumeInfo.total > 0 ? Math.round(volumeInfo.nonZero/volumeInfo.total*100) : 0}%)`);
      console.log('      DB Average:', volumeInfo.avgFromDB || 'N/A');
      if (volumeInfo.calculatedAvg) {
        console.log('      Calculated Average:', volumeInfo.calculatedAvg);
      }
      if (volumeInfo.firstNonZero) {
        console.log('      First non-zero:', JSON.stringify(volumeInfo.firstNonZero));
      } else {
        console.log('      ⚠️  All values are zero!');
      }

      console.log('\n' + '-'.repeat(60));
    }

    // Summary
    console.log('\n📌 SUMMARY:');
    console.log('If all values are zero, the issue is in data collection (frontend).');
    console.log('If values exist but charts are empty, the issue is in data display (charts).');

    await database.destroy();
  } catch (error) {
    console.error('❌ Error:', error);
    await database.destroy();
  }
}

// Run analysis
analyzeAudioData();