/**
 * Database Field Verification Script
 *
 * This script verifies that Azure Text Analytics data is properly
 * saved to the database and matches the expected C# format.
 *
 * Usage: node src/test/verifyDatabaseFields.js [sessionId]
 */

// Set environment to development
process.env.NODE_ENV = 'development';

const database = require('../database');

async function verifySession(sessionId) {
  try {
    console.log('\n====================================================');
    console.log('📊 DATABASE FIELD VERIFICATION');
    console.log('====================================================\n');

    // Fetch session from database
    const session = await database('presentation_session')
      .where({ id: sessionId })
      .first();

    if (!session) {
      console.error(`❌ Session with ID ${sessionId} not found`);
      return;
    }

    console.log(`✅ Session found: ${session.session_name}`);
    console.log(`📅 Created: ${session.recording_date}`);
    console.log('\n----------------------------------------');

    // Check Sentiment Scores
    console.log('\n🎭 SENTIMENT SCORES:');
    console.log(`  Positive: ${(session.sentiment_positive_score * 100).toFixed(2)}%`);
    console.log(`  Negative: ${(session.sentiment_negative_score * 100).toFixed(2)}%`);
    console.log(`  Neutral: ${(session.sentiment_neutral_score * 100).toFixed(2)}%`);

    // Verify scores sum to ~1.0 (within rounding error)
    const scoreSum = session.sentiment_positive_score +
                     session.sentiment_negative_score +
                     session.sentiment_neutral_score;
    console.log(`  Total: ${(scoreSum * 100).toFixed(2)}% ${Math.abs(scoreSum - 1.0) < 0.01 ? '✅' : '⚠️'}`);

    // Check Key Phrases
    console.log('\n🔑 KEY PHRASES:');
    if (session.key_phrases) {
      try {
        const keyPhrases = JSON.parse(session.key_phrases);
        console.log(`  Found ${keyPhrases.length} key phrases`);
        if (keyPhrases.length > 0) {
          console.log('  Sample phrases:');
          keyPhrases.slice(0, 5).forEach((phrase, i) => {
            console.log(`    ${i + 1}. ${phrase}`);
          });
        }
      } catch (e) {
        console.log('  ⚠️ Invalid JSON format');
      }
    } else {
      console.log('  ❌ No key phrases found');
    }

    // Check Transcript Sentences
    console.log('\n📝 TRANSCRIPT SENTENCES:');
    if (session.transcript_sentences) {
      try {
        const sentences = JSON.parse(session.transcript_sentences);
        console.log(`  Found ${sentences.length} analyzed sentences`);

        if (sentences.length > 0) {
          // Check structure of first sentence
          const firstSentence = sentences[0];
          console.log('\n  First sentence structure:');
          console.log(`    ✅ Has 'text': ${firstSentence.hasOwnProperty('text')}`);
          console.log(`    ✅ Has 'sentiment': ${firstSentence.hasOwnProperty('sentiment')}`);
          console.log(`    ✅ Has 'positiveScore': ${firstSentence.hasOwnProperty('positiveScore')}`);
          console.log(`    ✅ Has 'negativeScore': ${firstSentence.hasOwnProperty('negativeScore')}`);
          console.log(`    ✅ Has 'neutralScore': ${firstSentence.hasOwnProperty('neutralScore')}`);
          console.log(`    ✅ Has 'offset': ${firstSentence.hasOwnProperty('offset')}`);
          console.log(`    ✅ Has 'duration': ${firstSentence.hasOwnProperty('duration')}`);

          console.log('\n  Sample sentence:');
          console.log(`    Text: "${firstSentence.text?.substring(0, 50)}..."`);
          console.log(`    Sentiment: ${firstSentence.sentiment}`);
          console.log(`    Scores: Pos=${(firstSentence.positiveScore * 100).toFixed(1)}% Neg=${(firstSentence.negativeScore * 100).toFixed(1)}% Neu=${(firstSentence.neutralScore * 100).toFixed(1)}%`);
        }
      } catch (e) {
        console.log('  ⚠️ Invalid JSON format:', e.message);
      }
    } else {
      console.log('  ❌ No analyzed sentences found');
    }

    // Check other important fields
    console.log('\n📊 OTHER ANALYSIS DATA:');
    console.log(`  Transcript Text: ${session.transcript_text ? session.transcript_text.length + ' characters' : 'Empty'}`);
    console.log(`  Words per Minute: ${session.words_per_minute || 'N/A'}`);
    console.log(`  Overall Score: ${session.overall_score || 'N/A'}`);
    console.log(`  Filler Words: ${session.filler_words ? 'Present' : 'Missing'}`);

    console.log('\n====================================================');
    console.log('✅ VERIFICATION COMPLETE');
    console.log('====================================================\n');

  } catch (error) {
    console.error('❌ Error verifying session:', error);
  } finally {
    // Close database connection
    await database.destroy();
  }
}

async function listRecentSessions() {
  try {
    console.log('\n📋 RECENT SESSIONS:');
    console.log('----------------------------------------');

    const sessions = await database('presentation_session')
      .select('id', 'session_name', 'recording_date', 'overall_score')
      .orderBy('recording_date', 'desc')
      .limit(10);

    if (sessions.length === 0) {
      console.log('No sessions found in database');
      return;
    }

    sessions.forEach(session => {
      const date = new Date(session.recording_date).toLocaleString();
      console.log(`  ID: ${session.id} | ${session.session_name} | ${date} | Score: ${session.overall_score || 'N/A'}`);
    });

    console.log('\nUse: node src/test/verifyDatabaseFields.js [sessionId]');
    console.log('Example: node src/test/verifyDatabaseFields.js ' + sessions[0].id);

  } catch (error) {
    console.error('❌ Error listing sessions:', error);
  } finally {
    await database.destroy();
  }
}

// Main execution
const sessionId = process.argv[2];

if (sessionId) {
  verifySession(parseInt(sessionId));
} else {
  listRecentSessions();
}