/**
 * PRESENTATION PDF MIGRATION SCRIPT
 *
 * Adds file_blob_key column to presentation_pdf table and migrates existing data
 *
 * CHANGES:
 * - Adds file_blob_key column (nullable, will be required later)
 * - Converts existing file_url values to blob keys
 * - Preserves file_url for backward compatibility (will be removed later)
 *
 * SAFETY FEATURES:
 * - Dry-run mode (test without writing)
 * - Pre-migration verification
 * - Post-migration verification
 * - Comprehensive logging
 * - Zero data loss guarantee
 *
 * USAGE:
 *   node migrations/add-file-blob-key-to-presentations.js --dry-run  (Test mode)
 *   node migrations/add-file-blob-key-to-presentations.js            (Execute migration)
 */

require('module-alias/register');
const path = require('path');
const moduleAlias = require('module-alias');

// Setup module aliases
moduleAlias.addAliases({
  '@database': path.join(__dirname, '../src/installation/database.installation.js'),
});

const database = require('@database');

// Check if running in dry-run mode
const isDryRun = process.argv.includes('--dry-run');

// Logging utilities
const log = {
  info: (msg) => console.log(`[INFO] ${msg}`),
  success: (msg) => console.log(`[SUCCESS] ✓ ${msg}`),
  warning: (msg) => console.log(`[WARNING] ⚠ ${msg}`),
  error: (msg) => console.log(`[ERROR] ✗ ${msg}`),
  section: (msg) => console.log(`\n${'='.repeat(60)}\n${msg}\n${'='.repeat(60)}`),
};

/**
 * Extract blob key from Azure Blob Storage URL
 * Example: https://immblend.blob.core.windows.net/presentation-materials/pdf/user123-1234567890.pdf
 * Returns: pdf/user123-1234567890.pdf
 */
function extractBlobKeyFromUrl(fileUrl) {
  if (!fileUrl) return null;

  try {
    const url = new URL(fileUrl);
    // Split path and remove leading slash
    const pathParts = url.pathname.split('/');
    // First part is container name, rest is blob name
    // e.g., /presentation-materials/pdf/user123-1234567890.pdf
    // container: presentation-materials, blob: pdf/user123-1234567890.pdf
    if (pathParts.length >= 3) {
      // Remove first empty element and container name
      const blobKey = pathParts.slice(2).join('/');
      return blobKey;
    }

    log.warning(`Could not parse URL: ${fileUrl}`);
    return null;
  } catch (error) {
    log.error(`Invalid URL: ${fileUrl} - ${error.message}`);
    return null;
  }
}

/**
 * Check if file_blob_key column already exists
 */
async function checkColumnExists() {
  log.section('STEP 1: CHECKING SCHEMA');

  try {
    const hasColumn = await database.schema.hasColumn('presentation_pdf', 'file_blob_key');

    if (hasColumn) {
      log.warning('Column file_blob_key already exists in presentation_pdf table');
      return true;
    }

    log.info('Column file_blob_key does not exist yet');
    return false;
  } catch (error) {
    log.error(`Failed to check schema: ${error.message}`);
    throw error;
  }
}

/**
 * Add file_blob_key column to presentation_pdf table
 */
async function addColumn() {
  log.section('STEP 2: ADDING COLUMN');

  try {
    if (isDryRun) {
      log.info('[DRY-RUN] Would add column: file_blob_key VARCHAR(500) NULL');
      log.success('[DRY-RUN] Column would be added successfully');
      return;
    }

    await database.schema.table('presentation_pdf', (table) => {
      table.string('file_blob_key', 500).nullable().after('file_url');
    });

    log.success('Added column file_blob_key to presentation_pdf table');
  } catch (error) {
    log.error(`Failed to add column: ${error.message}`);
    throw error;
  }
}

/**
 * Get all presentations with file_url
 */
async function getPresentations() {
  log.section('STEP 3: FETCHING PRESENTATIONS');

  try {
    const presentations = await database('presentation_pdf')
      .select('id', 'ugid', 'presentation_name', 'file_url', 'file_blob_key')
      .orderBy('id');

    log.info(`Found ${presentations.length} presentations in database`);

    if (presentations.length > 0) {
      log.info('Sample presentation:');
      const sample = presentations[0];
      log.info(`  - ID: ${sample.id}`);
      log.info(`  - Name: ${sample.presentation_name}`);
      log.info(`  - file_url: ${sample.file_url}`);
      log.info(`  - file_blob_key: ${sample.file_blob_key || '(null)'}`);
    }

    return presentations;
  } catch (error) {
    log.error(`Failed to fetch presentations: ${error.message}`);
    throw error;
  }
}

/**
 * Migrate file_url to file_blob_key
 */
async function migrateBlobKeys(presentations) {
  log.section(`STEP 4: MIGRATING BLOB KEYS ${isDryRun ? '(DRY-RUN MODE)' : ''}`);

  const stats = {
    total: presentations.length,
    migrated: 0,
    skipped: 0,
    failed: 0,
    errors: []
  };

  for (const presentation of presentations) {
    try {
      // Skip if blob key already exists
      if (presentation.file_blob_key) {
        log.info(`Skipping presentation ${presentation.id}: blob key already exists`);
        stats.skipped++;
        continue;
      }

      // Skip if no file_url
      if (!presentation.file_url) {
        log.warning(`Skipping presentation ${presentation.id}: no file_url found`);
        stats.skipped++;
        continue;
      }

      // Extract blob key from URL
      const blobKey = extractBlobKeyFromUrl(presentation.file_url);

      if (!blobKey) {
        log.error(`Failed to extract blob key from URL for presentation ${presentation.id}`);
        stats.failed++;
        stats.errors.push({
          id: presentation.id,
          name: presentation.presentation_name,
          error: 'Could not extract blob key from URL'
        });
        continue;
      }

      if (isDryRun) {
        log.info(`[DRY-RUN] Would migrate presentation ${presentation.id}:`);
        log.info(`  → Name: ${presentation.presentation_name}`);
        log.info(`  → URL: ${presentation.file_url}`);
        log.info(`  → Blob Key: ${blobKey}`);
        stats.migrated++;
      } else {
        // Update database
        await database('presentation_pdf')
          .where({ id: presentation.id })
          .update({ file_blob_key: blobKey });

        log.success(`Migrated presentation ${presentation.id}: ${presentation.presentation_name}`);
        log.info(`  → Blob Key: ${blobKey}`);
        stats.migrated++;
      }

    } catch (error) {
      log.error(`Failed to migrate presentation ${presentation.id}: ${error.message}`);
      stats.failed++;
      stats.errors.push({
        id: presentation.id,
        name: presentation.presentation_name,
        error: error.message
      });
    }
  }

  return stats;
}

/**
 * Post-migration verification
 */
async function verifyMigration() {
  log.section('STEP 5: POST-MIGRATION VERIFICATION');

  try {
    if (isDryRun) {
      log.info('[DRY-RUN] Skipping verification (no actual changes made)');
      return;
    }

    // Count presentations with blob keys
    const withBlobKey = await database('presentation_pdf')
      .whereNotNull('file_blob_key')
      .count('* as count');

    const withoutBlobKey = await database('presentation_pdf')
      .whereNull('file_blob_key')
      .count('* as count');

    const withBlobKeyCount = withBlobKey[0].count;
    const withoutBlobKeyCount = withoutBlobKey[0].count;

    log.info(`Presentations with blob key: ${withBlobKeyCount}`);
    log.info(`Presentations without blob key: ${withoutBlobKeyCount}`);

    if (withoutBlobKeyCount > 0) {
      log.warning(`Found ${withoutBlobKeyCount} presentations without blob keys`);

      // Show which ones are missing
      const missing = await database('presentation_pdf')
        .whereNull('file_blob_key')
        .select('id', 'presentation_name', 'file_url');

      log.warning('Presentations without blob keys:');
      missing.forEach(p => {
        log.warning(`  - ID ${p.id}: ${p.presentation_name} (${p.file_url || 'no URL'})`);
      });
    } else {
      log.success('All presentations have blob keys');
    }

    // Verify blob keys are valid (not empty, follow pattern)
    const invalidBlobKeys = await database('presentation_pdf')
      .whereNotNull('file_blob_key')
      .andWhere('file_blob_key', '=', '')
      .count('* as count');

    if (invalidBlobKeys[0].count > 0) {
      log.error(`Found ${invalidBlobKeys[0].count} presentations with empty blob keys`);
      throw new Error('Invalid blob keys detected');
    }

    log.success('Post-migration verification passed');

  } catch (error) {
    log.error(`Verification failed: ${error.message}`);
    throw error;
  }
}

/**
 * Main migration function
 */
async function main() {
  log.section(`PRESENTATION MIGRATION SCRIPT - ${isDryRun ? 'DRY-RUN MODE' : 'EXECUTION MODE'}`);

  if (isDryRun) {
    log.warning('Running in DRY-RUN mode - NO changes will be made to the database');
  } else {
    log.warning('Running in EXECUTION mode - Changes WILL be written to the database');
  }

  try {
    // Step 1: Check if column exists
    const columnExists = await checkColumnExists();

    // Step 2: Add column if it doesn't exist
    if (!columnExists) {
      await addColumn();
    } else {
      log.info('Skipping column creation (already exists)');
    }

    // Step 3: Get all presentations
    const presentations = await getPresentations();

    // Step 4: Migrate blob keys
    const stats = await migrateBlobKeys(presentations);

    // Step 5: Verify migration
    await verifyMigration();

    // Final summary
    log.section('MIGRATION SUMMARY');
    log.info(`Total presentations: ${stats.total}`);
    log.success(`Successfully migrated: ${stats.migrated}`);
    log.warning(`Skipped: ${stats.skipped}`);
    if (stats.failed > 0) {
      log.error(`Failed: ${stats.failed}`);
      stats.errors.forEach(err => {
        log.error(`  - ID ${err.id} (${err.name}): ${err.error}`);
      });
    }

    if (isDryRun) {
      log.section('DRY-RUN COMPLETE');
      log.info('No changes were made to the database');
      log.info('To execute the migration, run:');
      log.info('  node migrations/add-file-blob-key-to-presentations.js');
    } else {
      log.section('MIGRATION COMPLETE');
      log.success('Database schema has been updated successfully');
      log.info('Next steps:');
      log.info('  1. Update controllers to use file_blob_key instead of file_url');
      log.info('  2. Create PDF streaming proxy endpoint');
      log.info('  3. Test thoroughly before removing file_url column');
    }

    process.exit(0);

  } catch (error) {
    log.section('MIGRATION FAILED');
    log.error(`Fatal error: ${error.message}`);
    log.error(error.stack);
    process.exit(1);

  } finally {
    // Cleanup
    if (database) {
      await database.destroy();
      log.info('Database connection closed');
    }
  }
}

// Run migration
main();
