/**
 * Multer Middleware for File Uploads
 *
 * Configures multer for handling multipart/form-data file uploads
 * with validation for file types and size limits
 */

const multer = require("multer");
const utils = require("@utils");

/**
 * File filter for image uploads
 * Only allows PNG and JPEG/JPG formats
 */
const imageFileFilter = (req, file, cb) => {
  const allowedMimeTypes = [
    "image/jpeg",
    "image/jpg",
    "image/png",
  ];

  if (allowedMimeTypes.includes(file.mimetype)) {
    cb(null, true);
  } else {
    cb(
      new Error(
        "Invalid image format. Only PNG and JPG/JPEG images are allowed."
      ),
      false
    );
  }
};

/**
 * File filter for PDF uploads
 * Only allows PDF documents
 */
const pdfFileFilter = (req, file, cb) => {
  if (file.mimetype === "application/pdf") {
    cb(null, true);
  } else {
    cb(
      new Error("Invalid file type. Only PDF files are allowed."),
      false
    );
  }
};

/**
 * File filter for video uploads
 * Only allows WebM and MP4 video formats
 * Handles MIME types with codecs (e.g., video/webm;codecs=vp8)
 */
const videoFileFilter = (req, file, cb) => {
  console.log('🎬 [videoFileFilter] Checking file:', {
    fieldname: file.fieldname,
    originalname: file.originalname,
    mimetype: file.mimetype,
    size: file.size,
  });

  // Check if MIME type starts with video/webm or video/mp4
  // This handles cases like "video/webm;codecs=vp8"
  const mimeType = file.mimetype.toLowerCase();
  const isValidVideo = mimeType.startsWith("video/webm") || mimeType.startsWith("video/mp4");

  if (isValidVideo) {
    console.log('✅ [videoFileFilter] Video format ACCEPTED:', mimeType);
    cb(null, true);
  } else {
    console.log('❌ [videoFileFilter] Video format REJECTED:', mimeType);
    cb(
      new Error(`Invalid video format: ${file.mimetype}. Only WebM and MP4 videos are allowed.`),
      false
    );
  }
};

/**
 * Multer configuration for profile picture uploads
 * - Memory storage (file stored in buffer)
 * - 5MB file size limit
 * - PNG and JPG/JPEG formats only
 */
const profilePictureUpload = multer({
  storage: multer.memoryStorage(),
  limits: {
    fileSize: 5 * 1024 * 1024, // 5MB max file size
  },
  fileFilter: imageFileFilter,
});

/**
 * Multer configuration for PDF presentation uploads
 * - Memory storage (file stored in buffer)
 * - 20MB file size limit
 * - PDF files only
 */
const pdfUpload = multer({
  storage: multer.memoryStorage(),
  limits: {
    fileSize: 20 * 1024 * 1024, // 20MB max file size
  },
  fileFilter: pdfFileFilter,
});

/**
 * Multer configuration for video session uploads
 * - Memory storage (file stored in buffer)
 * - 100MB file size limit
 * - WebM and MP4 formats only
 */
const videoUpload = multer({
  storage: multer.memoryStorage(),
  limits: {
    fileSize: 100 * 1024 * 1024, // 100MB max file size
  },
  fileFilter: videoFileFilter,
});

/**
 * Multer error handler middleware
 * Formats multer errors into standardized API responses
 */
const handleMulterError = (err, req, res, next) => {
  console.log('🚨 [handleMulterError] Called with:', {
    hasError: !!err,
    errorType: err ? err.constructor.name : 'None',
    errorMessage: err ? err.message : 'No error',
    errorCode: err ? err.code : 'N/A',
  });

  if (err instanceof multer.MulterError) {
    // Multer-specific errors
    console.log('❌ [handleMulterError] Multer error detected:', err.code);
    if (err.code === "LIMIT_FILE_SIZE") {
      return utils.responseSnippet(
        res,
        400,
        "File size exceeds the maximum limit",
        null
      );
    }
    if (err.code === "LIMIT_FILE_COUNT") {
      return utils.responseSnippet(
        res,
        400,
        "Too many files uploaded",
        null
      );
    }
    if (err.code === "LIMIT_UNEXPECTED_FILE") {
      return utils.responseSnippet(
        res,
        400,
        "Unexpected field name for file upload",
        null
      );
    }

    return utils.responseSnippet(res, 400, err.message, null);
  } else if (err) {
    // Custom errors (like file type validation)
    console.log('❌ [handleMulterError] Custom error:', err.message);
    return utils.responseSnippet(res, 400, err.message, null);
  }

  console.log('✅ [handleMulterError] No error, continuing...');
  next();
};

module.exports = {
  profilePictureUpload,
  pdfUpload,
  videoUpload,
  handleMulterError,
};
