/**
 * OpenAI Assistant Service for AI Feedback Generation
 *
 * This service integrates with OpenAI Assistants API to generate structured feedback
 * for presentation sessions based on analysis data.
 * Uses the same Assistant API approach as the C# backend.
 */

const axios = require('axios');
const config = require('@config');

class OpenAIAssistantService {
  constructor() {
    this.apiKey = config.openai?.apiKey || process.env.OPENAI_API_KEY;
    this.assistantId = config.openai?.assistantId || process.env.OPENAI_ASSISTANT_ID || 'asst_p7JnY1XhjEJqQNnoL50dWo7P';
    this.apiUrl = 'https://api.openai.com/v1';
    this.organization = config.openai?.organization || process.env.OPENAI_ORGANIZATION;
    this.projectId = config.openai?.projectId || process.env.OPENAI_PROJECT_ID;

    // Default headers for all requests
    this.headers = {
      'Authorization': `Bearer ${this.apiKey}`,
      'Content-Type': 'application/json',
      'OpenAI-Beta': 'assistants=v2'
    };

    if (this.organization) {
      this.headers['OpenAI-Organization'] = this.organization;
    }

    if (this.projectId) {
      this.headers['OpenAI-Project'] = this.projectId;
    }
  }

  /**
   * Generate structured feedback using OpenAI Assistant API
   * @param {Object} analysisData - Session analysis data
   * @param {string} language - Target language ('de' or 'en')
   * @returns {Promise<string>} JSON string containing structured feedback
   */
  async generateStructuredFeedbackAsync(analysisData, language = 'de') {
    if (!this.apiKey) {
      throw new Error('OpenAI API key is not configured');
    }

    const prompt = this.buildFeedbackPrompt(analysisData, language);

    try {
      // Step 1: Create a thread with the message
      console.log('Creating OpenAI thread...');
      const threadResponse = await axios.post(
        `${this.apiUrl}/threads`,
        {
          messages: [
            {
              role: 'user',
              content: prompt
            }
          ]
        },
        { headers: this.headers }
      );

      const threadId = threadResponse.data.id;
      console.log(`Thread created: ${threadId}`);

      // Step 2: Create a run with the assistant
      console.log('Creating run with assistant...');
      const runResponse = await axios.post(
        `${this.apiUrl}/threads/${threadId}/runs`,
        {
          assistant_id: this.assistantId
        },
        { headers: this.headers }
      );

      const runId = runResponse.data.id;
      console.log(`Run created: ${runId}`);

      // Step 3: Poll for run completion
      console.log('Waiting for assistant response...');
      let runStatus = 'in_progress';
      let attempts = 0;
      const maxAttempts = 60; // 60 seconds timeout

      while (runStatus === 'in_progress' || runStatus === 'queued') {
        if (attempts >= maxAttempts) {
          throw new Error('Timeout waiting for OpenAI assistant response');
        }

        await new Promise(resolve => setTimeout(resolve, 1000)); // Wait 1 second

        const statusResponse = await axios.get(
          `${this.apiUrl}/threads/${threadId}/runs/${runId}`,
          { headers: this.headers }
        );

        runStatus = statusResponse.data.status;
        console.log(`Run status: ${runStatus} (attempt ${attempts + 1})`);

        if (runStatus === 'failed' || runStatus === 'cancelled' || runStatus === 'expired') {
          throw new Error(`Run failed with status: ${runStatus}`);
        }

        attempts++;
      }

      // Step 4: Retrieve messages from the thread
      console.log('Retrieving assistant response...');
      const messagesResponse = await axios.get(
        `${this.apiUrl}/threads/${threadId}/messages`,
        { headers: this.headers }
      );

      // Find the assistant's response (most recent message from assistant)
      const assistantMessages = messagesResponse.data.data.filter(msg => msg.role === 'assistant');

      if (assistantMessages.length === 0) {
        throw new Error('No assistant response found');
      }

      // Extract text from the assistant's message
      const latestMessage = assistantMessages[0];
      let responseText = '';

      for (const content of latestMessage.content) {
        if (content.type === 'text') {
          responseText = content.text.value;
          break;
        }
      }

      if (!responseText) {
        throw new Error('No text content found in assistant response');
      }

      // Try to extract JSON from the response
      let jsonResponse = responseText;

      // If the response contains markdown code blocks, extract the JSON
      const jsonMatch = responseText.match(/```json\s*([\s\S]*?)\s*```/) ||
                       responseText.match(/```\s*([\s\S]*?)\s*```/) ||
                       responseText.match(/(\{[\s\S]*\})/);

      if (jsonMatch) {
        jsonResponse = jsonMatch[1] || jsonMatch[0];
      }

      // Validate that it's valid JSON
      try {
        JSON.parse(jsonResponse);
      } catch (e) {
        console.warn('Response is not valid JSON, returning as-is');
      }

      return jsonResponse;

    } catch (error) {
      console.error('OpenAI Assistant API Error:', error.response?.data || error.message);

      // Fallback to Chat Completions API if Assistant fails
      console.log('Falling back to Chat Completions API...');
      return await this.generateWithChatCompletion(analysisData, language);
    }
  }

  /**
   * Fallback method using Chat Completions API
   */
  async generateWithChatCompletion(analysisData, language) {
    const prompt = this.buildFeedbackPrompt(analysisData, language);

    const response = await axios.post(
      `${this.apiUrl}/chat/completions`,
      {
        model: 'gpt-3.5-turbo',
        messages: [
          {
            role: 'system',
            content: 'You are an expert presentation coach providing constructive feedback. You MUST respond with valid JSON only. Do not include any markdown formatting or code blocks. Return only the JSON object.'
          },
          {
            role: 'user',
            content: prompt
          }
        ],
        temperature: 0.7,
        max_tokens: 2500
      },
      {
        headers: {
          'Authorization': `Bearer ${this.apiKey}`,
          'Content-Type': 'application/json'
        }
      }
    );

    return response.data.choices[0].message.content;
  }

  /**
   * Build the feedback prompt with session data (same as C# backend)
   */
  buildFeedbackPrompt(context, language) {
    const fillerWordsText = context.fillerWords?.length > 0
      ? context.fillerWords.slice(0, 5).map(f => f.word || f).join(', ')
      : 'none';

    const targetLanguage = language === 'de' ? 'German (Deutsch)' : 'English';
    const languageExample = language === 'de'
      ? 'Example in German: "Sie haben eine sehr gute Präsentation gehalten. Ihre Sprechgeschwindigkeit war optimal."'
      : 'Example in English: "You delivered a very good presentation. Your speaking pace was optimal."';

    return `You are an expert presentation coach with 20+ years of experience in public speaking, communication training, and professional development. Your role is to provide detailed, constructive feedback on presentation performances.

=== CRITICAL INSTRUCTIONS - READ CAREFULLY ===

LANGUAGE REQUIREMENT:
- You MUST write ALL content in ${targetLanguage} language
- Every single word, sentence, and paragraph must be in ${targetLanguage}
- Do NOT mix languages under any circumstances
- ${languageExample}

OUTPUT FORMAT REQUIREMENT:
- Return ONLY valid JSON without any markdown formatting
- Do NOT use code blocks like \`\`\`json or \`\`\`
- Your response must be a valid JSON object that starts with { and ends with }
- Ensure all JSON strings are properly escaped
- Use double quotes for all JSON keys and string values

=== PRESENTATION SESSION DATA ===

SESSION OVERVIEW:
- Duration: ${context.sessionLength || 0} minutes
- Overall Performance Score: ${context.overallScore || 0}/100

SPEECH DELIVERY METRICS:
- Words per Minute (WPM): ${context.wordsPerMinute || 0} (optimal range: 140-160 WPM)
- Speech Speed Score: ${context.speechSpeedScore || 0}/100
- Speaking Clarity Score: ${context.speakingClearlyScore || 0}/100

PAUSE PATTERN ANALYSIS:
- Pauses per Minute: ${context.pausesPerMinute || 0}
- Total Pause Duration: ${context.totalPauseTime || 0} seconds
- Pause Quality Score: ${context.pausesScore || 0}/100

VOCAL CHARACTERISTICS:
- Average Pitch: ${context.averagePitch || 0} Hz
- Average Volume Level: ${context.averageVolume || 0} dB

CONTENT QUALITY METRICS:
- Sentiment Score: ${context.sentiment || 0} (scale: 0=negative, 1=positive)
- Filler Words Count: ${context.fillerWordCount || 0} occurrences (examples: ${fillerWordsText})
- Filler Words Score: ${context.fillerWordsScore || 0}/100
- Repetition Count: ${context.repetitionCount || 0}
- Repetition Score: ${context.repetitionScore || 0}/100

FULL PRESENTATION TRANSCRIPTION:
${context.transcriptionText || ''}

=== YOUR TASK AS A PRESENTATION COACH ===

As an experienced presentation coach, you need to analyze the presentation performance data above and provide detailed, actionable feedback. Your analysis should be:

1. CONSTRUCTIVE: Focus on improvement while acknowledging strengths
2. SPECIFIC: Reference actual metrics and provide concrete examples
3. ACTIONABLE: Give clear, practical advice the presenter can implement
4. COMPREHENSIVE: Cover all aspects of presentation delivery and content
5. ENCOURAGING: Maintain a supportive, professional tone throughout

ANALYSIS REQUIREMENTS:

1. TECHNICAL DELIVERY ANALYSIS:
   - Evaluate speech speed (optimal: 140-160 WPM)
   - Assess pause patterns and their effectiveness
   - Analyze filler word usage and its impact
   - Review voice modulation (pitch and volume variations)

2. PRESENTATION STRUCTURE EVALUATION:
   - Introduction: Was it engaging? Did it set clear expectations?
   - Body: Was the content well-organized with clear transitions?
   - Conclusion: Did it summarize key points and leave a lasting impression?
   - Overall flow: Was the logical progression easy to follow?

3. ARGUMENTATION & CONTENT QUALITY:
   - Are claims supported with evidence?
   - Is the reasoning logical and coherent?
   - Does the content match the intended message?
   - Is the language appropriate for the audience?

4. COMMUNICATION EFFECTIVENESS:
   - Clarity of message delivery
   - Engagement techniques used
   - Professional language usage
   - Overall impact and persuasiveness

IMPORTANT: All your feedback text MUST be written in ${targetLanguage}. This is critical for the user's understanding.

JSON RESPONSE STRUCTURE (FOLLOW EXACTLY):

{
  "summary": "Provide a 2-3 sentence executive summary in ${targetLanguage}. Include: (1) Overall performance level with reference to the ${context.overallScore || 0}/10 score, (2) The most significant strength observed, (3) The primary area needing improvement. Be balanced and professional.",

  "strengths": [
    "First major strength: Be specific and reference actual data/metrics from the presentation in ${targetLanguage}",
    "Second major strength: Highlight another positive aspect with concrete examples in ${targetLanguage}",
    "Third major strength: Mention additional strength that contributes to effectiveness in ${targetLanguage}"
  ],

  "areasForImprovement": [
    "First improvement area: Identify specific issue with actionable solution in ${targetLanguage}",
    "Second improvement area: Point out another challenge with practical advice in ${targetLanguage}",
    "Third improvement area: Suggest additional refinement for professional growth in ${targetLanguage}"
  ],

  "presentationStructure": "Write a detailed analysis (200-300 words) in ${targetLanguage} covering: INTRODUCTION - Did it capture attention? Was the purpose clear? Were objectives stated? BODY - How was information organized? Were transitions smooth? Did examples support main points? Was there a logical progression? CONCLUSION - Were key points summarized? Was there a clear call-to-action or takeaway? Did it leave a strong impression? OVERALL COHERENCE - How well did all parts work together? Provide 2-3 specific suggestions for structural improvements with examples of how to implement them.",

  "argumentationPlausibility": "Write a thorough evaluation (200-300 words) in ${targetLanguage} analyzing: CLAIMS - Were assertions clear and specific? EVIDENCE - Was supporting data/examples credible and relevant? LOGIC - Did arguments follow sound reasoning? Were there any logical fallacies? PERSUASIVENESS - How convincing was the overall argument? IMPROVEMENTS - Provide 2-3 specific recommendations to strengthen argumentation, such as adding data, clarifying logic chains, or addressing counterarguments. Include examples of how to enhance credibility.",

  "recommendations": [
    {
      "category": "${language === 'de' ? 'Sprechgeschwindigkeit' : 'Speech Speed'}",
      "observation": "Write detailed observation in ${targetLanguage}: The presenter spoke at ${context.wordsPerMinute || 0} words per minute. Compare this to the optimal range of 140-160 WPM. Explain how this speed affected comprehension and engagement. Was it too fast (causing confusion), too slow (causing boredom), or just right?",
      "suggestion": "Provide 2-3 specific techniques in ${targetLanguage} to achieve optimal speech speed. For example: practice with a metronome, use deliberate pauses, record yourself, or practice tongue twisters. Be specific about HOW to implement these techniques.",
      "metrics": {
        "currentValue": ${context.wordsPerMinute || 0},
        "optimalValue": 150.0,
        "unit": "WPM"
      }
    },
    {
      "category": "${language === 'de' ? 'Pausen' : 'Pauses'}",
      "observation": "Write detailed observation in ${targetLanguage}: The presentation had ${context.pausesPerMinute || 0} pauses per minute with an average duration of ${(context.totalPauseTime / (context.pausesPerMinute * context.sessionLength * 60) || 0).toFixed(1)} seconds. Explain whether pauses were used effectively for emphasis, transitions, or audience processing time. Were they too frequent (disrupting flow) or too rare (creating monotony)?",
      "suggestion": "Provide specific guidance in ${targetLanguage} on strategic pause usage: Where to pause (after key points, before transitions), how long to pause (1-3 seconds for emphasis, 2-4 for transitions), and practice techniques (mark pause points in script, breathe consciously).",
      "metrics": {
        "currentValue": ${context.pausesPerMinute || 0},
        "optimalValue": 2.0,
        "unit": "per minute"
      }
    },
    {
      "category": "${language === 'de' ? 'Füllwörter' : 'Filler Words'}",
      "observation": "Write detailed observation in ${targetLanguage}: Detected ${context.fillerWordCount || 0} filler words in the presentation${context.fillerWords?.length > 0 ? ' including: ' + fillerWordsText : ''}. Explain how these affected professional delivery and audience perception. Calculate frequency (filler words per minute) and impact on message clarity.",
      "suggestion": "Provide practical strategies in ${targetLanguage} to eliminate filler words: (1) Pause instead of filling silence, (2) Slow down speaking pace to think ahead, (3) Practice with recording to build awareness, (4) Use transition phrases instead of 'um' or 'uh'. Include specific exercises.",
      "metrics": {
        "currentValue": ${context.fillerWordCount || 0},
        "optimalValue": 0,
        "unit": "occurrences"
      }
    }
  ],

  "closingMessage": "Write an encouraging, personalized closing message (3-4 sentences) in ${targetLanguage}. Reference their specific score of ${context.overallScore || 0}/10 and acknowledge their effort. Highlight their strongest area and express confidence in their ability to improve. End with motivational words for their next presentation. Make it warm and professional."
}

=== FINAL CRITICAL REMINDERS ===

1. Your ENTIRE response must be valid JSON - no exceptions
2. ALL text content must be in ${targetLanguage}
3. Properly escape any quotes inside JSON strings using backslash (\")
4. Do NOT include line breaks inside JSON string values - use spaces instead
5. Ensure all arrays have at least the specified number of items
6. Double-check that your JSON is valid before responding
7. Your response should be parseable by JSON.parse() without any errors

Remember: You are providing feedback to help someone improve their presentation skills. Be thorough, specific, and constructive while maintaining a supportive tone throughout.`;
  }

  /**
   * Extract analysis data from session for feedback generation
   */
  extractAnalysisData(session) {
    // Calculate filler words count
    let fillerWordCount = 0;
    let fillerWords = [];
    try {
      if (session.filler_words) {
        fillerWords = JSON.parse(session.filler_words);
        fillerWordCount = fillerWords.length;
      }
    } catch (e) {
      console.warn('Failed to parse filler_words:', e);
    }

    // Calculate sentiment
    const sentiment = session.sentiment_positive_score || 0;

    return {
      sessionLength: Math.round(session.session_length / 60), // Convert to minutes
      overallScore: session.overall_score || 0,
      wordsPerMinute: session.words_per_minute || 0,
      speechSpeedScore: session.speech_speed_score || 0,
      speakingClearlyScore: session.speaking_clearly_score || 0,
      pausesPerMinute: session.pauses_per_minute || 0,
      totalPauseTime: session.total_pause_time || 0,
      pausesScore: session.pauses_score || 0,
      averagePitch: session.avg_pitch || 0,
      averageVolume: session.avg_volume || 0,
      sentiment: sentiment / 100, // Convert to 0-1 scale
      fillerWordCount: fillerWordCount,
      fillerWords: fillerWords,
      fillerWordsScore: session.filler_words_score || 0,
      repetitionCount: session.repetition_count || 0,
      repetitionScore: session.repetition_score || 0,
      transcriptionText: session.transcript_text || ''
    };
  }
}

module.exports = new OpenAIAssistantService();