// Imports
const WebSocket = require('ws');
const { setupSpeechWebSocket } = require('../services/speech/speechWebSocket');

// Initialize WebSocket server
const setupWebSocket = (server) => {
  console.log('🔧 [WebSocket] Setting up WebSocket server...');

  // Create WebSocket server instance (noServer mode for multiple paths)
  const wss = new WebSocket.Server({ noServer: true });

  // Setup HTTP upgrade handler for different WebSocket paths
  server.on('upgrade', (request, socket, head) => {
    const url = new URL(request.url, `ws://${request.headers.host}`);
    const pathname = url.pathname;

    console.log('🔌 [WebSocket] Upgrade request for path:', pathname);

    // Route to appropriate WebSocket handler
    if (pathname.startsWith('/ws/speech')) {
      // Speech recognition WebSocket
      wss.handleUpgrade(request, socket, head, (ws) => {
        wss.emit('connection', ws, request);
      });
    } else if (pathname === '/ws') {
      // General WebSocket (legacy/future use)
      wss.handleUpgrade(request, socket, head, (ws) => {
        wss.emit('connection', ws, request);
      });
    } else {
      console.warn('⚠️  [WebSocket] Unknown WebSocket path:', pathname);
      socket.destroy();
    }
  });

  // Setup speech recognition WebSocket handler
  setupSpeechWebSocket(wss);

  // Handle general WebSocket connections (ping/pong, etc.)
  wss.on("connection", function(socket, request) {
    const url = new URL(request.url, `ws://${request.headers.host}`);
    const pathname = url.pathname;

    // Only handle general /ws path here (speech is handled by setupSpeechWebSocket)
    if (pathname !== '/ws') {
      return;
    }

    console.log('🔌 [WebSocket] General WebSocket connection established');

    socket.isAlive = true;

    // Handle messages
    socket.on("message", function(message) {
      try {
        const data = JSON.parse(message);
        if (data.type === "ping") {
          socket.send(JSON.stringify({
            type: "pong",
            timestamp: Date.now()
          }));
        }
      } catch (error) {
        console.error('❌ [WebSocket] Invalid message:', error);
      }
    });

    // Handle errors
    socket.on("error", (error) => {
      console.error('❌ [WebSocket] Socket error:', error);
      socket.isAlive = false;
      socket.terminate();
    });

    // Handle disconnections
    socket.on("close", () => {
      console.log('🔌 [WebSocket] General connection closed');
      socket.isAlive = false;
    });
  });

  console.log('✅ [WebSocket] WebSocket server initialized');
  console.log('   📍 General WebSocket: /ws');
  console.log('   📍 Speech Recognition: /ws/speech');

  return wss;
};

module.exports = setupWebSocket;