/**
 * Migration Script: Move video_blob_key to presentation_video_blobs table
 *
 * This script:
 * 1. Creates the new presentation_video_blobs table if not exists
 * 2. Migrates existing video_blob_key data from presentation_session
 * 3. Keeps the original column for backward compatibility (can be removed later)
 *
 * Usage: node migrations/migrate-video-blobs.js
 */

// Set environment to development
process.env.NODE_ENV = 'development';

// Setup module aliases
require('module-alias/register');
const database = require('@database');

async function migrateVideoBlobs() {
  try {
    console.log('\n====================================================');
    console.log('🔄 MIGRATING VIDEO BLOBS TO SEPARATE TABLE');
    console.log('====================================================\n');

    // Step 1: Check if the new table exists
    const tableExists = await database.schema.hasTable('presentation_video_blobs');

    if (!tableExists) {
      console.log('📦 Creating presentation_video_blobs table...');

      await database.schema.createTable('presentation_video_blobs', (t) => {
        t.increments('id');
        t.integer('session_id').unsigned().unique().notNullable();
        t.string('video_blob_key', 500).notNullable();
        t.integer('blob_size').nullable();
        t.string('content_type', 100).nullable();
        t.timestamp('uploaded_at').defaultTo(database.fn.now());
        t.string('storage_provider', 50).defaultTo('azure');

        // Add foreign key constraint
        t.foreign('session_id').references('id').inTable('presentation_session').onDelete('CASCADE');

        // Add index for faster lookups
        t.index('session_id');
      });

      console.log('✅ Table created successfully');
    } else {
      console.log('✅ Table already exists');
    }

    // Step 2: Check if video_blob_key column still exists in presentation_session
    const hasColumn = await database.schema.hasColumn('presentation_session', 'video_blob_key');

    if (hasColumn) {
      // Step 3: Migrate existing data
      console.log('\n📊 Checking for existing video blob keys to migrate...');

      const sessions = await database('presentation_session')
        .select('id', 'session_name', 'video_blob_key')
        .whereNotNull('video_blob_key')
        .where('video_blob_key', '!=', '');

      console.log(`Found ${sessions.length} sessions with video blob keys`);

      let migrated = 0;
      let skipped = 0;

      for (const session of sessions) {
        // Check if already migrated
        const existing = await database('presentation_video_blobs')
          .where('session_id', session.id)
          .first();

        if (existing) {
          console.log(`⏭️ Session ${session.id} already migrated`);
          skipped++;
          continue;
        }

        // Insert into new table
        await database('presentation_video_blobs').insert({
          session_id: session.id,
          video_blob_key: session.video_blob_key,
          content_type: 'video/mp4', // Default, can be updated later
          storage_provider: 'azure'
        });

        console.log(`✅ Migrated session ${session.id}: ${session.session_name}`);
        migrated++;
      }

      console.log('\n====================================================');
      console.log(`✅ MIGRATION COMPLETE`);
      console.log(`   Migrated: ${migrated} sessions`);
      console.log(`   Skipped: ${skipped} sessions (already migrated)`);
      console.log(`   Total: ${sessions.length} sessions`);
      console.log('====================================================\n');

      console.log('ℹ️ Note: The video_blob_key column in presentation_session table');
      console.log('   has been kept for backward compatibility.');
      console.log('   It can be removed in a future migration after updating all code.');
    } else {
      console.log('⚠️ video_blob_key column not found in presentation_session table');
      console.log('   (It may have been removed already)');
    }

    // Step 4: Verify migration
    console.log('\n🔍 VERIFICATION:');
    const videoBlobs = await database('presentation_video_blobs').count('* as count');
    console.log(`Total records in presentation_video_blobs: ${videoBlobs[0].count}`);

    // Sample data
    const samples = await database('presentation_video_blobs')
      .select('*')
      .limit(3);

    if (samples.length > 0) {
      console.log('\nSample records:');
      samples.forEach(blob => {
        console.log(`  Session ${blob.session_id}: ${blob.video_blob_key.substring(0, 50)}...`);
      });
    }

    await database.destroy();

  } catch (error) {
    console.error('❌ Migration failed:', error);
    await database.destroy();
    process.exit(1);
  }
}

// Execute migration
migrateVideoBlobs();