const database = require("@database");
const config = require("@config");
const utils = require("@utils");
const nodemailer = require("nodemailer");

const codeGeneratorAuthentication = async (req, res) => {
  try {
    const userInfo = {
      ip: req.ip,
      agent: req.get("user-agent"),
      email: req.body.email,
      language: req.body.language || 'en', // 'en' or 'de'
    };

    if (config.adminSettings.isAdminBased) {
      const existingUser = await database("user_data")
        .where({ email: userInfo.email })
        .first();

      if (!existingUser) {
        return utils.responseSnippet(
          res,
          403,
          "This email is not registered. Please contact the administrator.",
          null
        );
      }
    }

    const recentAttempts = await database("user_login_code")
      .where({ email: userInfo.email })
      .where(
        "time_generated",
        ">",
        database.raw("NOW() - INTERVAL ? MINUTE", [
          config.limits.spam.requetsRateLimitInterval,
        ])
      )
      .count("id as count")
      .first();

    if (
      recentAttempts &&
      recentAttempts.count >= config.limits.spam.requestsRateLimit
    ) {
      return utils.responseSnippet(
        res,
        429,
        `Too many requests. Please try again in ${config.limits.spam.requetsRateLimitInterval} minute(s).`,
        null
      );
    }

    const generateVerificationCode = () => {
      const codeLength = config.verification.codeLength || 6;
      const useLetters = config.verification.includeLetters || false;

      if (useLetters) {
        const chars = "0123456789ABCDEFGHIJKLMNOPQRSTUVWXYZ";
        let code = "";
        for (let i = 0; i < codeLength; i++) {
          code += chars.charAt(
            Math.floor(Math.random() * chars.length)
          );
        }
        return code;
      } else {
        const min = Math.pow(10, codeLength - 1);
        const max = Math.pow(10, codeLength) - 1;
        return Math.floor(
          min + Math.random() * (max - min + 1)
        ).toString();
      }
    };

    const verificationCode = generateVerificationCode();

    await database("user_login_code").insert({
      link: verificationCode,
      email: userInfo.email,
      agent: userInfo.agent,
      ip: userInfo.ip,
      language: userInfo.language,
      time_generated: database.fn.now(),
    });

    console.log('='.repeat(60));
    console.log(`📧 EMAIL VERIFICATION REQUEST`);
    console.log(`Email: ${userInfo.email}`);
    console.log(`Code: ${verificationCode}`);
    console.log(`IP: ${userInfo.ip}`);
    console.log(`Time: ${new Date().toISOString()}`);
    console.log('='.repeat(60));

    // Create email transporter
    console.log(`🔧 Creating email transporter...`);
    console.log(`Host: ${config.email.host}`);
    console.log(`Port: ${config.email.port}`);
    console.log(`User: ${config.email.user}`);
    console.log(`Secure: ${config.email.secure}`);

    const transporter = nodemailer.createTransport({
      host: config.email.host,
      port: config.email.port,
      secure: config.email.secure,
      auth: {
        user: config.email.user,
        pass: config.email.pass,
      },
    });

    // Send verification email in selected language
    console.log(`📨 Sending email to ${userInfo.email} in language: ${userInfo.language}...`);

    const emailContent = userInfo.language === 'de' ? {
      subject: "Ihr Verifizierungscode | KommunikationsTrainer",
      text: `Hallo,
Willkommen bei KommunikationsTrainer!

Ihr Verifizierungscode lautet: ${verificationCode}
Dieser Code ist aus Sicherheitsgründen zeitlich begrenzt gültig.

Mit freundlichen Grüßen
Ihr KommunikationsTrainer-Team`,
      html: `<!DOCTYPE html>
<html lang="de">
<head>
  <meta charset="UTF-8">
  <meta name="viewport" content="width=device-width, initial-scale=1.0">
  <title>Ihr Verifizierungscode</title>
  <!--[if mso]>
  <noscript>
    <xml>
      <o:OfficeDocumentSettings>
        <o:PixelsPerInch>96</o:PixelsPerInch>
      </o:OfficeDocumentSettings>
    </xml>
  </noscript>
  <![endif]-->
</head>
<body style="margin: 0; padding: 0; font-family: -apple-system, BlinkMacSystemFont, 'Segoe UI', Roboto, 'Helvetica Neue', Arial, sans-serif; background-color: #f5f5f5;">
  <div style="width: 100%; background-color: #f5f5f5; padding: 20px 0;">
    <table cellpadding="0" cellspacing="0" border="0" width="100%" style="max-width: 600px; margin: 0 auto; background-color: #ffffff; border-radius: 12px; overflow: hidden; box-shadow: 0 2px 8px rgba(0,0,0,0.05);">
      <!-- Header -->
      <tr>
        <td style="background: linear-gradient(135deg, #667eea 0%, #764ba2 100%); padding: 40px 20px; text-align: center;">
          <h1 style="color: #ffffff; font-size: 28px; font-weight: 600; margin: 0; letter-spacing: -0.5px;">KommunikationsTrainer</h1>
          <p style="color: rgba(255,255,255,0.9); font-size: 14px; margin: 10px 0 0 0;">Ihre Präsentations-Trainingsplattform</p>
        </td>
      </tr>

      <!-- Content -->
      <tr>
        <td style="padding: 40px 30px;">
          <h2 style="color: #333333; font-size: 22px; font-weight: 600; margin: 0 0 20px 0;">Hallo!</h2>
          <p style="color: #555555; font-size: 16px; line-height: 1.6; margin: 0 0 25px 0;">
            Willkommen bei KommunikationsTrainer!
          </p>

          <!-- Verification Code Box -->
          <div style="background: linear-gradient(135deg, #667eea 0%, #764ba2 100%); border-radius: 8px; padding: 25px; text-align: center; margin: 30px 0;">
            <p style="color: rgba(255,255,255,0.9); font-size: 14px; margin: 0 0 15px 0; text-transform: uppercase; letter-spacing: 1px;">
              Ihr Verifizierungscode
            </p>
            <div style="background-color: #ffffff; border-radius: 6px; padding: 15px 25px; display: inline-block;">
              <span style="color: #764ba2; font-size: 32px; font-weight: 700; letter-spacing: 3px; font-family: 'Courier New', monospace;">
                ${verificationCode}
              </span>
            </div>
          </div>

          <p style="color: #777777; font-size: 14px; line-height: 1.6; margin: 25px 0; text-align: center; font-style: italic;">
            Dieser Code ist aus Sicherheitsgründen zeitlich begrenzt gültig.
          </p>

          <!-- CTA Button -->
          <div style="text-align: center; margin: 35px 0;">
            <a href="#" style="display: inline-block; background: linear-gradient(135deg, #667eea 0%, #764ba2 100%); color: #ffffff; text-decoration: none; padding: 14px 40px; border-radius: 6px; font-size: 16px; font-weight: 600; letter-spacing: 0.5px;">
              Zur Anmeldung
            </a>
          </div>

          <p style="color: #555555; font-size: 16px; line-height: 1.6; margin: 30px 0 0 0;">
            Mit freundlichen Grüßen,<br>
            <strong>Ihr KommunikationsTrainer-Team</strong>
          </p>
        </td>
      </tr>

      <!-- Footer -->
      <tr>
        <td style="background-color: #f8f9fa; padding: 25px 30px; border-top: 1px solid #e9ecef;">
          <p style="color: #999999; font-size: 12px; line-height: 1.6; margin: 0; text-align: center;">
            Diese E-Mail wurde automatisch generiert. Bitte antworten Sie nicht auf diese Nachricht.<br>
            © 2025 KommunikationsTrainer. Alle Rechte vorbehalten.
          </p>
        </td>
      </tr>
    </table>
  </div>
</body>
</html>`
    } : {
      subject: "Your Verification Code | KommunikationsTrainer",
      text: `Hello,
Welcome to KommunikationsTrainer!

Your verification code is: ${verificationCode}
This code is valid for a limited time for security reasons.

Best regards,
Your KommunikationsTrainer Team`,
      html: `<!DOCTYPE html>
<html lang="en">
<head>
  <meta charset="UTF-8">
  <meta name="viewport" content="width=device-width, initial-scale=1.0">
  <title>Your Verification Code</title>
  <!--[if mso]>
  <noscript>
    <xml>
      <o:OfficeDocumentSettings>
        <o:PixelsPerInch>96</o:PixelsPerInch>
      </o:OfficeDocumentSettings>
    </xml>
  </noscript>
  <![endif]-->
</head>
<body style="margin: 0; padding: 0; font-family: -apple-system, BlinkMacSystemFont, 'Segoe UI', Roboto, 'Helvetica Neue', Arial, sans-serif; background-color: #f5f5f5;">
  <div style="width: 100%; background-color: #f5f5f5; padding: 20px 0;">
    <table cellpadding="0" cellspacing="0" border="0" width="100%" style="max-width: 600px; margin: 0 auto; background-color: #ffffff; border-radius: 12px; overflow: hidden; box-shadow: 0 2px 8px rgba(0,0,0,0.05);">
      <!-- Header -->
      <tr>
        <td style="background: linear-gradient(135deg, #667eea 0%, #764ba2 100%); padding: 40px 20px; text-align: center;">
          <h1 style="color: #ffffff; font-size: 28px; font-weight: 600; margin: 0; letter-spacing: -0.5px;">KommunikationsTrainer</h1>
          <p style="color: rgba(255,255,255,0.9); font-size: 14px; margin: 10px 0 0 0;">Your Presentation Training Platform</p>
        </td>
      </tr>

      <!-- Content -->
      <tr>
        <td style="padding: 40px 30px;">
          <h2 style="color: #333333; font-size: 22px; font-weight: 600; margin: 0 0 20px 0;">Hello!</h2>
          <p style="color: #555555; font-size: 16px; line-height: 1.6; margin: 0 0 25px 0;">
            Welcome to KommunikationsTrainer!
          </p>

          <!-- Verification Code Box -->
          <div style="background: linear-gradient(135deg, #667eea 0%, #764ba2 100%); border-radius: 8px; padding: 25px; text-align: center; margin: 30px 0;">
            <p style="color: rgba(255,255,255,0.9); font-size: 14px; margin: 0 0 15px 0; text-transform: uppercase; letter-spacing: 1px;">
              Your Verification Code
            </p>
            <div style="background-color: #ffffff; border-radius: 6px; padding: 15px 25px; display: inline-block;">
              <span style="color: #764ba2; font-size: 32px; font-weight: 700; letter-spacing: 3px; font-family: 'Courier New', monospace;">
                ${verificationCode}
              </span>
            </div>
          </div>

          <p style="color: #777777; font-size: 14px; line-height: 1.6; margin: 25px 0; text-align: center; font-style: italic;">
            This code is valid for a limited time for security reasons.
          </p>

          <!-- CTA Button -->
          <div style="text-align: center; margin: 35px 0;">
            <a href="#" style="display: inline-block; background: linear-gradient(135deg, #667eea 0%, #764ba2 100%); color: #ffffff; text-decoration: none; padding: 14px 40px; border-radius: 6px; font-size: 16px; font-weight: 600; letter-spacing: 0.5px;">
              Go to Login
            </a>
          </div>

          <p style="color: #555555; font-size: 16px; line-height: 1.6; margin: 30px 0 0 0;">
            Best regards,<br>
            <strong>Your KommunikationsTrainer Team</strong>
          </p>
        </td>
      </tr>

      <!-- Footer -->
      <tr>
        <td style="background-color: #f8f9fa; padding: 25px 30px; border-top: 1px solid #e9ecef;">
          <p style="color: #999999; font-size: 12px; line-height: 1.6; margin: 0; text-align: center;">
            This email was generated automatically. Please do not reply to this message.<br>
            © 2025 KommunikationsTrainer. All rights reserved.
          </p>
        </td>
      </tr>
    </table>
  </div>
</body>
</html>`
    };
    // Send the actual email
    const emailInfo = await transporter.sendMail({
      from: `"KommunikationsTrainer" <${config.email.user}>`,
      to: userInfo.email,
      subject: emailContent.subject,
      text: emailContent.text,
      html: emailContent.html,
    });

    console.log(`✅ Email sent successfully!`);
    console.log(`Message ID: ${emailInfo.messageId}`);
    console.log(`Response: ${emailInfo.response}`);
    console.log('='.repeat(60));

    utils.responseSnippet(
      res,
      200,
      "Verification code has been sent to your email.",
      null
    );
  } catch (error) {
    console.log('='.repeat(60));
    console.error("❌ EMAIL SENDING ERROR");
    console.error("Error type:", error.name);
    console.error("Error message:", error.message);
    console.error("Error code:", error.code);
    if (error.command) {
      console.error("SMTP Command:", error.command);
    }
    if (error.response) {
      console.error("SMTP Response:", error.response);
    }
    console.error("Full error:", error);
    console.log('='.repeat(60));
    utils.catchErrorHandlerSnippet(res, error, __dirname);
  }
};

module.exports = codeGeneratorAuthentication;
