/**
 * Create Session Controller
 *
 * Creates a new presentation session with video upload and complete analysis data
 * - Uploads video to Azure Blob Storage
 * - Stores video blob key (not URL) in database
 * - Saves all real-time analysis data from frontend
 * - Sets analysis_status as 'completed' (all analysis done in frontend)
 */

const database = require("@database");
const utils = require("@utils");
const azureStorage = require("@utils/azure.storage.util");
const { calculateAllScores } = require("@utils/score.calculation.util");
const textAnalyticsService = require("@services/textAnalytics.service");
const audioAnalysisService = require("@services/audioAnalysis.service");

const createSession = async (req, res) => {
  try {
    // Get user UGID from token
    const { ugid } = req.user;

    // Get video file from multer
    const videoFile = req.file;
    if (!videoFile) {
      return utils.responseSnippet(res, 400, "Video file is required", null);
    }

    // Parse JSON fields from request body and prepare for database storage
    // MySQL TEXT columns require JSON to be stringified, not objects
    const parseAndStringifyJSON = (field, defaultValue = null) => {
      try {
        if (!field) return defaultValue;
        // Parse the JSON string from FormData
        const parsed = JSON.parse(field);
        // Stringify it again for MySQL TEXT column storage
        return JSON.stringify(parsed);
      } catch (error) {
        console.error(`Error parsing JSON field:`, error);
        return defaultValue;
      }
    };

    // Extract basic session data
    const {
      session_name,
      // presentation_id, // REMOVED
      // locale, // REMOVED
      language, // Language code from frontend (de-DE, en-US)
      session_length,

      // Slide data (JSON) - REMOVED
      // slides,
      // slide_timestamps,
      // heatmap,
      // averages,

      // Transcript data
      transcript_text,
      transcript_words,
      transcript_sentences,
      transcribed_segments,
      filler_words,
      key_phrases,
      key_words,
      words_per_minute,

      // Sentiment scores
      sentiment_positive_score,
      sentiment_negative_score,
      sentiment_neutral_score,

      // Audio analysis data (JSON)
      audio_pitch_data,
      audio_volume_data,
      audio_pauses_data,
      // audio_timed_averages, // REMOVED

      // Audio analysis metrics
      avg_pitch,
      avg_volume,
      avg_pause_time,
      pauses_per_minute,
      total_pause_time,

      // Session scores
      speech_speed_score,
      pauses_score,
      repetition_score,
      filler_words_score,
      speaking_clearly_score,
      overall_score,
    } = req.body;

    console.log('📝 Creating session:', {
      session_name,
      ugid,
      videoSize: videoFile.size,
      videoType: videoFile.mimetype
    });

    // DEBUG: Log all received fields
    console.log('📦 Received req.body fields:', Object.keys(req.body));
    console.log('📊 Sample data check:', {
      hasTranscriptText: !!transcript_text,
      transcriptTextLength: transcript_text?.length || 0,
      hasTranscriptWords: !!transcript_words,
      hasSegments: !!transcribed_segments,
      hasAudioPitchData: !!audio_pitch_data,
      hasAudioVolumeData: !!audio_volume_data,
      sessionLength: session_length,
      wpm: words_per_minute,
    });

    // DEBUG: Log score values
    console.log('🎯 Score values received:', {
      speech_speed_score: speech_speed_score,
      pauses_score: pauses_score,
      repetition_score: repetition_score,
      filler_words_score: filler_words_score,
      speaking_clearly_score: speaking_clearly_score,
      overall_score: overall_score,
    });

    // First, insert session to get session ID
    const [sessionId] = await database("presentation_session").insert({
      ugid,
      session_name: session_name || "Untitled Session",
      // presentation_id: presentation_id || null, // REMOVED
      // locale: locale || "en", // REMOVED
      recording_date: new Date(),
      // status: "active", // REMOVED
      // analysis_status: "pending", // REMOVED
      // created_at: new Date(), // REMOVED
      // updated_at: new Date(), // REMOVED
    });

    console.log('✅ Session created with ID:', sessionId);

    // Upload video to Azure Blob Storage
    const timestamp = Date.now();
    const blobName = `video/${ugid}-${sessionId}-${timestamp}.webm`;
    const containerName = "presentation-materials";

    console.log('📤 Uploading video to Azure:', blobName);

    const uploadResult = await azureStorage.uploadFile(
      containerName,
      blobName,
      videoFile.buffer,
      {
        contentType: videoFile.mimetype,
        metadata: {
          ugid,
          sessionId: sessionId.toString(),
          uploadedAt: new Date().toISOString(),
        },
      }
    );

    if (!uploadResult.success) {
      // Rollback session creation if video upload fails
      await database("presentation_session").where({ id: sessionId }).delete();
      return utils.responseSnippet(
        res,
        500,
        "Failed to upload video to storage",
        null
      );
    }

    console.log('✅ Video uploaded successfully:', blobName);

    // ============================================================================
    // AUDIO ANALYSIS (Backend Processing)
    // ============================================================================

    let backendPitchData = [];
    let backendVolumeData = [];
    let backendAvgPitch = 0;
    let backendAvgVolume = 0;

    // Check what data frontend provided
    console.log('🎵 Audio analysis check:', {
      hasFrontendPitch: !!audio_pitch_data,
      hasFrontendVolume: !!audio_volume_data,
      frontendPitchSamples: audio_pitch_data ? audio_pitch_data.length : 0,
      frontendVolumeSamples: audio_volume_data ? audio_volume_data.length : 0,
      willUseBackendAnalysis: !audio_pitch_data || !audio_volume_data || audio_pitch_data.length === 0 || audio_volume_data.length === 0
    });

    // If frontend didn't provide pitch/volume data or data is empty, analyze on backend
    if (!audio_pitch_data || !audio_volume_data || audio_pitch_data.length === 0 || audio_volume_data.length === 0) {
      console.log('🎵 Frontend audio data missing or empty, running backend analysis...');
      try {
        const audioAnalysisResult = await audioAnalysisService.analyzeFromBuffer(
          videoFile.buffer,
          videoFile.originalname,
          60000 // 60 second timeout
        );

        // Only use backend data if we got valid results
        if (audioAnalysisResult.pitchData && audioAnalysisResult.pitchData.length > 0) {
          backendPitchData = audioAnalysisResult.pitchData;
          console.log(`✅ Backend pitch analysis: ${backendPitchData.length} samples`);
        }

        if (audioAnalysisResult.volumeData && audioAnalysisResult.volumeData.length > 0) {
          backendVolumeData = audioAnalysisResult.volumeData;
          console.log(`✅ Backend volume analysis: ${backendVolumeData.length} samples`);
        }

        backendAvgPitch = audioAnalysisResult.avgPitch || 0;
        backendAvgVolume = audioAnalysisResult.avgVolume || 0;

        // If backend analysis produced no results, log a warning
        if (backendPitchData.length === 0 && backendVolumeData.length === 0) {
          console.warn('⚠️ Backend audio analysis produced no results');
        }
      } catch (error) {
        console.error('❌ Backend audio analysis failed:', error.message);
        // Continue without audio analysis data
      }
    } else {
      console.log('✅ Using frontend audio data, skipping backend analysis');
    }

    // ============================================================================
    // CALCULATE SESSION SCORES
    // ============================================================================
    console.log('🧮 Calculating session scores...');

    // Prepare data for score calculation
    const parsedTranscriptWords = transcript_words ? JSON.parse(transcript_words) : [];
    const parsedFillerWords = filler_words ? JSON.parse(filler_words) : [];
    const parsedSegments = transcribed_segments ? JSON.parse(transcribed_segments) : [];

    // Calculate filler words count
    const fillerWordsCount = parsedFillerWords.reduce((sum, fw) => sum + (fw.count || 0), 0);

    // Calculate average confidence from segments
    const averageConfidence = parsedSegments.length > 0
      ? parsedSegments.reduce((sum, seg) => sum + (seg.confidence || 0), 0) / parsedSegments.length
      : 0;

    // Prepare lookup words for repetition calculation
    // For now, we don't have full lookup word data, so repetition score will default to 10
    const lookupWords = []; // TODO: Implement lookup word detection if needed

    const analysisData = {
      wordsPerMinute: parseFloat(words_per_minute) || 0,
      pausesPerMinute: parseFloat(pauses_per_minute) || 0,
      fillerWordsCount: fillerWordsCount,
      sessionLengthMinutes: (parseFloat(session_length) || 0) / 60, // Convert seconds to minutes
      averageConfidence: averageConfidence,
      lookupWords: lookupWords,
    };

    console.log('📊 Analysis data for scoring:', {
      wordsPerMinute: analysisData.wordsPerMinute,
      pausesPerMinute: analysisData.pausesPerMinute,
      fillerWordsCount: analysisData.fillerWordsCount,
      sessionLengthMinutes: analysisData.sessionLengthMinutes,
      averageConfidence: analysisData.averageConfidence,
    });

    // Calculate all scores
    const calculatedScores = calculateAllScores(analysisData);

    console.log('✅ Calculated scores:', calculatedScores);

    // ============================================================================
    // AZURE TEXT ANALYTICS
    // ============================================================================
    // Extract language code for Azure (use first 2 chars: 'de' or 'en')
    // Default to 'de' for German as per the system's primary use case
    const languageCode = language ? language.substring(0, 2) : 'de';

    console.log('🌐 Language for text analytics:', languageCode);
    console.log('📝 Transcript text length:', transcript_text?.length || 0);

    // Initialize variables for Azure Text Analytics results
    let sentimentScores = {
      sentiment_positive_score: 0.33,
      sentiment_negative_score: 0.33,
      sentiment_neutral_score: 0.34,
    };
    let extractedKeyPhrases = [];
    let extractedKeyWords = [];
    let analyzedSentences = [];

    // Only call Azure if we have transcript text
    if (transcript_text && transcript_text.trim().length > 0) {
      try {
        console.log('🔍 Calling Azure Text Analytics...');

        // 1. Analyze sentiment (returns averaged scores across all chunks)
        const sentimentResult = await textAnalyticsService.analyzeSentiment(
          transcript_text,
          languageCode
        );

        sentimentScores = {
          sentiment_positive_score: sentimentResult.positiveScore,
          sentiment_negative_score: sentimentResult.negativeScore,
          sentiment_neutral_score: sentimentResult.neutralScore,
        };

        console.log('✅ Sentiment scores from Azure:', sentimentScores);

        // 2. Extract key phrases
        extractedKeyPhrases = await textAnalyticsService.extractKeyPhrases(
          transcript_text,
          languageCode
        );

        console.log(`✅ Extracted ${extractedKeyPhrases.length} key phrases from Azure`);

        // 3. Extract key words (individual important words, not phrases)
        extractedKeyWords = textAnalyticsService.extractKeyWords(
          transcript_text,
          languageCode
        );

        console.log(`✅ Extracted ${extractedKeyWords.length} key words after filtering stop words`);

        // 4. Analyze sentences individually (for transcript_sentences field)
        // Parse transcribed_segments if it's a string
        let segments = [];
        try {
          segments = transcribed_segments ? JSON.parse(transcribed_segments) : [];
        } catch (e) {
          // Already an object/array
          segments = transcribed_segments || [];
        }

        analyzedSentences = await textAnalyticsService.analyzeSentences(
          transcript_text,
          segments,
          languageCode
        );

        console.log(`✅ Analyzed ${analyzedSentences.length} sentences with sentiment`);

      } catch (error) {
        console.error('❌ Azure Text Analytics error:', error);
        console.error('⚠️ Using fallback values for sentiment scores');
        // Keep default values if Azure fails
      }
    } else {
      console.log('⚠️ No transcript text available, using default sentiment scores');
    }

    // Prepare update data
    const updateData = {
        // Video blob key moved to presentation_video_blobs table
        // video_blob_key: blobName, // Now stored in separate table

        // Session metadata
        session_length: session_length || null,

        // Slide data (JSON) - REMOVED
        // slides: parseAndStringifyJSON(slides),
        // slide_timestamps: parseAndStringifyJSON(slide_timestamps),
        // heatmap: parseAndStringifyJSON(heatmap),
        // averages: parseAndStringifyJSON(averages),

        // Transcript data
        transcript_text: transcript_text || null,
        transcript_words: parseAndStringifyJSON(transcript_words),
        // Use Azure-analyzed sentences if available, otherwise use frontend data
        transcript_sentences: analyzedSentences.length > 0
          ? JSON.stringify(analyzedSentences)
          : parseAndStringifyJSON(transcript_sentences),
        transcribed_segments: parseAndStringifyJSON(transcribed_segments),
        filler_words: parseAndStringifyJSON(filler_words),
        // Use Azure-extracted key phrases if available, otherwise use frontend data
        key_phrases: extractedKeyPhrases.length > 0
          ? JSON.stringify(extractedKeyPhrases)
          : parseAndStringifyJSON(key_phrases),
        // Use extracted key words if available, otherwise use frontend data
        key_words: extractedKeyWords.length > 0
          ? JSON.stringify(extractedKeyWords)
          : parseAndStringifyJSON(key_words),
        words_per_minute: words_per_minute || null,

        // Sentiment scores (from Azure Text Analytics or defaults if failed)
        sentiment_positive_score: sentimentScores.sentiment_positive_score,
        sentiment_negative_score: sentimentScores.sentiment_negative_score,
        sentiment_neutral_score: sentimentScores.sentiment_neutral_score,

        // Audio analysis data (JSON)
        // Use backend analysis if frontend didn't provide data
        audio_pitch_data: audio_pitch_data
          ? parseAndStringifyJSON(audio_pitch_data)
          : backendPitchData.length > 0
            ? JSON.stringify(backendPitchData)
            : null,
        audio_volume_data: audio_volume_data
          ? parseAndStringifyJSON(audio_volume_data)
          : backendVolumeData.length > 0
            ? JSON.stringify(backendVolumeData)
            : null,
        audio_pauses_data: parseAndStringifyJSON(audio_pauses_data),
        // audio_timed_averages: parseAndStringifyJSON(audio_timed_averages), // REMOVED

        // Audio analysis metrics (use backend values if frontend didn't provide)
        avg_pitch: avg_pitch || backendAvgPitch || null,
        avg_volume: avg_volume || backendAvgVolume || null,
        avg_pause_time: avg_pause_time || null,
        pauses_per_minute: pauses_per_minute || null,
        total_pause_time: total_pause_time || null,

        // Session scores (CALCULATED in backend, not from frontend!)
        speech_speed_score: calculatedScores.speechSpeedScore,
        pauses_score: calculatedScores.pausesScore,
        repetition_score: calculatedScores.repetitionScore,
        filler_words_score: calculatedScores.fillerWordsScore,
        speaking_clearly_score: calculatedScores.speakingClearlyScore,
        overall_score: calculatedScores.overallScore,

        // Mark analysis as completed (all analysis done in frontend) - REMOVED
        // analysis_status: "completed",
        // updated_at: new Date(),
    };

    // DEBUG: Log what we're about to save
    console.log('💾 Updating session with data:', {
      video_blob_key: updateData.video_blob_key,
      session_length: updateData.session_length,
      transcript_text_length: updateData.transcript_text?.length || 0,
      has_transcript_words: !!updateData.transcript_words,
      has_segments: !!updateData.transcribed_segments,
      has_audio_pitch: !!updateData.audio_pitch_data,
      has_audio_volume: !!updateData.audio_volume_data,
      wpm: updateData.words_per_minute,
      analysis_status: updateData.analysis_status,
    });

    // Update session with all analysis data
    await database("presentation_session")
      .where({ id: sessionId })
      .update(updateData);

    console.log('✅ Session updated with analysis data');

    // Insert video blob key into the new table
    // First check if it already exists (for idempotency)
    const existingVideoBlob = await database("presentation_video_blobs")
      .where({ session_id: sessionId })
      .first();

    if (existingVideoBlob) {
      // Update existing record
      await database("presentation_video_blobs")
        .where({ session_id: sessionId })
        .update({
          video_blob_key: blobName,
          blob_size: videoFile.buffer ? videoFile.buffer.length : null,
          content_type: videoFile.mimetype || 'video/mp4',
          uploaded_at: new Date()
        });
      console.log('✅ Video blob record updated');
    } else {
      // Insert new record
      await database("presentation_video_blobs").insert({
        session_id: sessionId,
        video_blob_key: blobName,
        blob_size: videoFile.buffer ? videoFile.buffer.length : null,
        content_type: videoFile.mimetype || 'video/mp4',
        storage_provider: 'azure'
      });
      console.log('✅ Video blob record created');
    }

    // Fetch the complete session data to verify what was saved
    const session = await database("presentation_session")
      .where({ id: sessionId })
      .first();

    // Fetch video blob info from the new table
    const videoBlob = await database("presentation_video_blobs")
      .where({ session_id: sessionId })
      .first();

    // DEBUG: Log what was actually saved in DB
    console.log('🔍 Verification - What was saved in DB:', {
      id: session.id,
      session_name: session.session_name,
      video_blob_key: videoBlob ? videoBlob.video_blob_key : 'Not in new table',
      session_length: session.session_length,
      transcript_text_length: session.transcript_text?.length || 0,
      transcript_words: session.transcript_words ? 'Has data' : 'NULL',
      transcribed_segments: session.transcribed_segments ? 'Has data' : 'NULL',
      audio_pitch_data: session.audio_pitch_data ? 'Has data' : 'NULL',
      audio_volume_data: session.audio_volume_data ? 'Has data' : 'NULL',
      words_per_minute: session.words_per_minute,
      analysis_status: session.analysis_status,
    });

    // DEBUG: Log score values saved in DB
    console.log('🎯 Scores saved in DB:', {
      speech_speed_score: session.speech_speed_score,
      pauses_score: session.pauses_score,
      repetition_score: session.repetition_score,
      filler_words_score: session.filler_words_score,
      speaking_clearly_score: session.speaking_clearly_score,
      overall_score: session.overall_score,
    });

    // Return only essential fields (not all JSON data)
    const responseData = {
      id: session.id,
      ugid: session.ugid,
      session_name: session.session_name,
      // locale: session.locale, // REMOVED
      // presentation_id: session.presentation_id, // REMOVED
      recording_date: session.recording_date,
      session_length: session.session_length,
      // status: session.status, // REMOVED
      // analysis_status: session.analysis_status, // REMOVED
      video_blob_key: videoBlob ? videoBlob.video_blob_key : null,
      overall_score: session.overall_score,
      words_per_minute: session.words_per_minute,
      // created_at: session.created_at, // REMOVED
      // updated_at: session.updated_at, // REMOVED
    };

    console.log('📊 Session created successfully:', {
      id: responseData.id,
      session_name: responseData.session_name,
      video_blob_key: responseData.video_blob_key,
      analysis_status: responseData.analysis_status,
    });

    return utils.responseSnippet(
      res,
      201,
      "Session created successfully",
      responseData
    );
  } catch (error) {
    console.error("❌ Create session error:", error);
    return utils.catchErrorHandlerSnippet(res, error, __dirname);
  }
};

module.exports = createSession;
