/**
 * Stream Session Video Controller (Proxy)
 *
 * Handles secure access to video files stored in Azure Blob Storage
 * - Validates authorization (user must own the session)
 * - Retrieves video_blob_key from database
 * - Streams video directly from Azure with support for range requests (seeking)
 */

const database = require("@database");
const utils = require("@utils");
const azureStorage = require("@utils/azure.storage.util");

const streamSessionVideo = async (req, res) => {
  try {
    console.log('🎥 === VIDEO FILE REQUEST ===');
    console.log('  Requested Session ID:', req.params.id);
    console.log('  Has req.user?', !!req.user);
    console.log('  Requesting User UGID:', req.user?.ugid || 'None');
    console.log('  Auth Header:', req.headers.authorization ? 'Present ✓' : 'Missing ✗');
    console.log('============================');

    const { id: sessionId } = req.params;
    const requestingUserUgid = req.user ? req.user.ugid : null;

    // Validate session ID
    if (!sessionId) {
      return utils.responseSnippet(res, 400, "Session ID is required", null);
    }

    // Authorization check - user must be authenticated
    if (!requestingUserUgid) {
      return utils.responseSnippet(
        res,
        401,
        "Authentication required to view video files",
        null
      );
    }

    // Get session details from database
    const session = await database("presentation_session")
      .where({ id: sessionId })
      .select("id", "ugid", "session_name")
      .first();

    if (!session) {
      return utils.responseSnippet(res, 404, "Session not found", null);
    }

    // Authorization check - user must own the session
    if (session.ugid !== requestingUserUgid) {
      return utils.responseSnippet(
        res,
        403,
        "You are not authorized to view this session video",
        null
      );
    }

    // Get video blob key from the new table
    const videoBlob = await database("presentation_video_blobs")
      .where({ session_id: sessionId })
      .select("video_blob_key", "blob_size", "content_type")
      .first();

    // Check if blob key exists
    if (!videoBlob || !videoBlob.video_blob_key) {
      return utils.responseSnippet(
        res,
        404,
        "Video file not found - blob key missing",
        null
      );
    }

    const containerName = "presentation-materials";
    const blobName = videoBlob.video_blob_key;

    console.log('📦 Streaming video from Azure:', {
      container: containerName,
      blob: blobName,
    });

    // Stream file from Azure Blob Storage (proxy without buffering)
    const streamResult = await azureStorage.streamFile(
      containerName,
      blobName
    );

    if (!streamResult.success) {
      return utils.responseSnippet(
        res,
        404,
        `Video file not found: ${streamResult.message}`,
        null
      );
    }

    // Set response headers for video
    res.setHeader("Content-Type", streamResult.contentType || "video/webm");
    res.setHeader("Content-Length", streamResult.contentLength);
    res.setHeader("Accept-Ranges", "bytes"); // Enable seeking
    res.setHeader("Cache-Control", "public, max-age=3600"); // Cache for 1 hour

    // Pipe Azure stream directly to client (no memory buffering!)
    streamResult.stream.pipe(res);

    console.log('✅ Video stream started successfully');
  } catch (error) {
    console.error("❌ Stream video file error:", error);

    // Check if headers were already sent
    if (!res.headersSent) {
      return utils.catchErrorHandlerSnippet(res, error, __dirname);
    }
  }
};

module.exports = streamSessionVideo;
