// Imports
const database = require("./database.installation");

// Create presentation tables and columns
const createPresentationTablesAndCols = async () => {
  try {
    await database.raw("SELECT 1");
    await createTablesAndCols();
  } catch (error) {
    console.error("Presentation tables initialization error:", error);
    throw error;
  }
};

// Create tables and columns
const createTablesAndCols = async () => {
  for (const table of tablesWithColumns) {
    try {
      const exists = await database.schema.hasTable(table.name);

      if (!exists) {
        await database.schema.createTable(table.name, (t) => {
          console.log(`* TABLE CREATION : ${table.name} was created`);

          table.columns.forEach((column) => {
            const columnName = Object.keys(column)[0];
            const columnDef = column[columnName];

            const parts = columnDef
              .replace("t.", "")
              .match(/(\w+)\('([^']+)'(?:,\s*'?(\w+)'?)?(?:,\s*(\d+))?\)(.+)?/);

            if (parts) {
              let [, type, name, param1, param2, modifiers = ""] = parts;

              let col;
              switch (type) {
                case "increments":
                  col = t.increments(name);
                  break;
                case "varchar":
                  col = t.string(name, param1 || 255);
                  break;
                case "text":
                  col = t.text(name, param1 || "text");
                  break;
                case "timestamp":
                  col = t.timestamp(name);
                  break;
                case "datetime":
                  col = t.datetime(name);
                  break;
                case "integer":
                  col = t.integer(name);
                  break;
                case "double":
                  col = t.double(name);
                  break;
                default:
                  col = t[type](name);
              }

              if (modifiers.includes(".unsigned()")) {
                col.unsigned();
              }
              if (modifiers.includes(".unique()")) {
                col.unique();
              }
              if (modifiers.includes(".nullable()")) {
                col.nullable();
              }
              if (modifiers.includes(".defaultTo('en')")) {
                col.defaultTo("en");
              }
              if (modifiers.includes(".defaultTo('active')")) {
                col.defaultTo("active");
              }
              if (modifiers.includes(".defaultTo('pending')")) {
                col.defaultTo("pending");
              }
              if (modifiers.includes("defaultTo(database.fn.now())")) {
                col.defaultTo(database.fn.now());
              }
            }

            console.log(`- COLUMN CREATION : ${columnName} added to ${table.name} table`);
          });
        });
      } else {
        for (const column of table.columns) {
          const columnName = Object.keys(column)[0];
          await addColumnIfNotExists(table.name, columnName, column[columnName]);
        }
      }
    } catch (error) {
      console.error(`Error processing table ${table.name}:`, error);
      throw error;
    }
  }
};

// Add column if not exists
const addColumnIfNotExists = async (tableName, columnName, columnDef) => {
  try {
    const hasColumn = await database.schema.hasColumn(tableName, columnName);
    if (!hasColumn) {
      await database.schema.alterTable(tableName, (t) => {
        const parts = columnDef
          .replace("t.", "")
          .match(/(\w+)\('([^']+)'(?:,\s*'?(\w+)'?)?(?:,\s*(\d+))?\)(.+)?/);

        if (parts) {
          let [, type, , param1, param2, modifiers = ""] = parts;

          let col;
          switch (type) {
            case "increments":
              col = t.increments(columnName);
              break;
            case "varchar":
              col = t.string(columnName, param1 || 255);
              break;
            case "text":
              col = t.text(columnName, param1 || "text");
              break;
            case "timestamp":
              col = t.timestamp(columnName);
              break;
            case "datetime":
              col = t.datetime(columnName);
              break;
            case "integer":
              col = t.integer(columnName);
              break;
            case "double":
              col = t.double(columnName);
              break;
            default:
              col = t[type](columnName);
          }

          if (modifiers.includes(".unsigned()")) {
            col.unsigned();
          }
          if (modifiers.includes(".unique()")) {
            col.unique();
          }
          if (modifiers.includes(".nullable()")) {
            col.nullable();
          }
          if (modifiers.includes(".defaultTo('en')")) {
            col.defaultTo("en");
          }
          if (modifiers.includes(".defaultTo('active')")) {
            col.defaultTo("active");
          }
          if (modifiers.includes(".defaultTo('pending')")) {
            col.defaultTo("pending");
          }
          if (modifiers.includes("defaultTo(database.fn.now())")) {
            col.defaultTo(database.fn.now());
          }
        }
        console.log(`- COLUMN CREATION : ${columnName} added to ${tableName}`);
      });
    }
  } catch (error) {
    console.error(`Error adding column ${columnName} to ${tableName}:`, error);
    throw error;
  }
};

// Tables with columns definition
const tablesWithColumns = [
  {
    name: "presentation_pdf",
    columns: [
      { id: "t.increments('id')" },
      { ugid: "t.varchar('ugid', 36)" },
      { presentation_name: "t.varchar('presentation_name', 255)" },
      { file_blob_key: "t.varchar('file_blob_key', 500)" },
    ],
  },
  {
    name: "presentation_session",
    columns: [
      // Primary identifiers
      { id: "t.increments('id')" },
      { ugid: "t.varchar('ugid', 36)" },
      { session_name: "t.varchar('session_name', 255)" },
      // { locale: "t.varchar('locale', 10).defaultTo('en')" }, // REMOVED: Not needed

      // Session metadata
      // { presentation_id: "t.integer('presentation_id').unsigned().nullable()" }, // REMOVED: Not needed
      { recording_date: "t.datetime('recording_date')" },
      { session_length: "t.double('session_length').nullable()" },
      // { status: "t.varchar('status', 50).defaultTo('active')" }, // REMOVED: Not needed
      // { analysis_status: "t.varchar('analysis_status', 50).defaultTo('pending')" }, // REMOVED: Not needed

      // Media file URLs
      // { audio_url: "t.varchar('audio_url', 1000).nullable()" }, // REMOVED: Deprecated - audio in video track
      { video_url: "t.varchar('video_url', 1000).nullable()" },

      // Media blob keys moved to presentation_video_blobs table
      // { video_blob_key: "t.varchar('video_blob_key', 500).nullable()" }, // MOVED to presentation_video_blobs

      // Slide data (JSON)
      // { slides: "t.text('slides', 'longtext').nullable()" }, // REMOVED: Not needed
      // { slide_timestamps: "t.text('slide_timestamps', 'longtext').nullable()" }, // REMOVED: Not needed
      // { heatmap: "t.text('heatmap', 'longtext').nullable()" }, // REMOVED: Not used
      // { averages: "t.text('averages', 'longtext').nullable()" }, // REMOVED: Not used

      // Transcript data
      { transcript_text: "t.text('transcript_text', 'longtext').nullable()" },
      { transcript_words: "t.text('transcript_words', 'longtext').nullable()" },
      { transcript_sentences: "t.text('transcript_sentences', 'longtext').nullable()" },
      { transcribed_segments: "t.text('transcribed_segments', 'longtext').nullable()" },
      { filler_words: "t.text('filler_words', 'longtext').nullable()" },
      { key_phrases: "t.text('key_phrases', 'longtext').nullable()" },
      { key_words: "t.text('key_words', 'longtext').nullable()" },
      { words_per_minute: "t.double('words_per_minute').nullable()" },

      // Sentiment analysis scores
      { sentiment_positive_score: "t.double('sentiment_positive_score').nullable()" },
      { sentiment_negative_score: "t.double('sentiment_negative_score').nullable()" },
      { sentiment_neutral_score: "t.double('sentiment_neutral_score').nullable()" },

      // Audio analysis data (JSON)
      { audio_pitch_data: "t.text('audio_pitch_data', 'longtext').nullable()" },
      { audio_volume_data: "t.text('audio_volume_data', 'longtext').nullable()" },
      { audio_pauses_data: "t.text('audio_pauses_data', 'longtext').nullable()" },
      // { audio_timed_averages: "t.text('audio_timed_averages', 'longtext').nullable()" }, // REMOVED: Not used

      // Audio analysis metrics
      { avg_pitch: "t.double('avg_pitch').nullable()" },
      { avg_volume: "t.double('avg_volume').nullable()" },
      { avg_pause_time: "t.double('avg_pause_time').nullable()" },
      { pauses_per_minute: "t.double('pauses_per_minute').nullable()" },
      { total_pause_time: "t.double('total_pause_time').nullable()" },

      // Session scores
      { speech_speed_score: "t.double('speech_speed_score').nullable()" },
      { pauses_score: "t.double('pauses_score').nullable()" },
      { repetition_score: "t.double('repetition_score').nullable()" },
      { filler_words_score: "t.double('filler_words_score').nullable()" },
      { speaking_clearly_score: "t.double('speaking_clearly_score').nullable()" },
      { overall_score: "t.double('overall_score').nullable()" },

      // Timestamps
      // { created_at: "t.timestamp('created_at').defaultTo(database.fn.now())" }, // REMOVED: Not needed
      // { updated_at: "t.timestamp('updated_at').defaultTo(database.fn.now())" }, // REMOVED: Not needed
    ],
  },
  {
    name: "presentation_feedback",
    columns: [
      { id: "t.increments('id')" },
      { session_id: "t.integer('session_id').unsigned()" },
      { feedback_text: "t.text('feedback_text', 'longtext')" },
      { language: "t.varchar('language', 10).defaultTo('en')" },
      { model_version: "t.varchar('model_version', 100).nullable()" },
      { created_at: "t.timestamp('created_at').defaultTo(database.fn.now())" },
      { updated_at: "t.timestamp('updated_at').defaultTo(database.fn.now())" },
    ],
  },
  {
    name: "presentation_video_blobs",
    columns: [
      { id: "t.increments('id')" },
      { session_id: "t.integer('session_id').unsigned().unique()" }, // One video per session
      { video_blob_key: "t.varchar('video_blob_key', 500)" },
      { blob_size: "t.integer('blob_size').nullable()" }, // Size in bytes
      { content_type: "t.varchar('content_type', 100).nullable()" }, // e.g., 'video/mp4'
      { uploaded_at: "t.timestamp('uploaded_at').defaultTo(database.fn.now())" },
      { storage_provider: "t.varchar('storage_provider', 50).defaultTo('azure')" }, // azure, aws, etc.
    ],
  },
];

module.exports = createPresentationTablesAndCols;
