const swaggerJsdoc = require("swagger-jsdoc");
const swaggerUi = require("swagger-ui-express");
const config = require("@config");
const fs = require("fs");
const path = require("path");

// Load all document files
const loadSwaggerDocuments = () => {
  const documents = {};
  const rootDir = process.env.NODE_ENV === "production" ? "./dist" : "./src";
  const routesPath = path.join(__dirname, "../api/documents");

  try {
    const routeFiles = fs.readdirSync(routesPath);
    console.log("Found route files:", routeFiles);

    routeFiles.forEach((file) => {
      if (file.endsWith(".documents.js")) {
        const moduleName = file.replace(".documents.js", "");
        const documentPath = path.join(routesPath, file);
        console.log(`Loading Swagger document: ${documentPath}`);
        documents[moduleName] = require(documentPath);
      }
    });

    console.log("Loaded documents:", Object.keys(documents));
  } catch (error) {
    console.error("Error loading Swagger documents:", error);
    console.error("Error details:", error.message);
    console.error("Attempted path:", path.join(__dirname, "../api/routes"));
  }

  return documents;
};

// Swagger definition
const swaggerOptions = {
  definition: {
    openapi: "3.0.0",
    info: {
      title: `${config.projectName} API Documentation`,
      version: "1.0.0",
      description: "API documentation for all endpoints",
      contact: {
        name: "Development Team",
        email: "info@starterkit.com",
      },
    },
    servers: [
      {
        url: `http://${config.server.host}:${config.server.port}`,
        description: "Development server",
      },
    ],
    components: {
      securitySchemes: {
        bearerAuth: {
          type: "http",
          scheme: "bearer",
          bearerFormat: "JWT",
        },
      },
    },
    paths: {},
  },
  apis: [], // No files to scan as we'll manually add paths
};

// Create paths from documents
const generatePaths = (docs) => {
  const paths = {};

  Object.keys(docs).forEach((module) => {
    const moduleDocs = docs[module];

    Object.keys(moduleDocs).forEach((endpoint) => {
      const pathKey = `/v1/${module}/${endpoint}`;
      paths[pathKey] = moduleDocs[endpoint];
    });
  });

  console.log("Generated API paths:", Object.keys(paths));
  return paths;
};

// Function to setup Swagger in the Express app
const setupSwagger = (app) => {
  console.log("Setting up Swagger documentation...");
  const documents = loadSwaggerDocuments();
  swaggerOptions.definition.paths = generatePaths(documents);

  const swaggerSpec = swaggerJsdoc(swaggerOptions);

  // Serve Swagger documentation
  app.use("/doc", swaggerUi.serve, swaggerUi.setup(swaggerSpec));

  // Serve Swagger JSON
  app.get("/doc.json", (req, res) => {
    res.setHeader("Content-Type", "application/json");
    res.send(swaggerSpec);
  });

  console.log(`Swagger documentation available at /doc`);
};

module.exports = setupSwagger;
